//
// Copyright (C) 1988-1999 Altera Corporation
//
// Any megafunction design, and related net list (encrypted or decrypted),
// support information, device programming or simulation file, and any
// other associated documentation or information provided by Altera or a
// partner under Altera's Megafunction Partnership Program may be used only
// to program PLD devices (but not masked PLD devices) from Altera.  Any
// other use of such megafunction design, net list, support information,
// device programming or simulation file, or any other related
// documentation or information is prohibited for any other purpose,
// including, but not limited to modification, reverse engineering, de-
// compiling, or use with any other silicon devices, unless such use is
// explicitly licensed under a separate agreement with Altera or a
// megafunction partner.  Title to the intellectual property, including
// patents, copyrights, trademarks, trade secrets, or maskworks, embodied
// in any such megafunction design, net list, support information, device
// programming or simulation file, or any other related documentation or
// information provided by Altera or a megafunction partner, remains with
// Altera, the megafunction partner, or their respective licensors.  No
// other licenses, including any licenses needed under any third party's
// intellectual property, are provided herein.
//--------------------------------------------------------------------------
//--------------------------------------------------------------------------
// Version 1.01 Date: 08/23/00 for MAX+plus II
//
// 08/22/00: Added synchronous mode to dcfifo.
// 08/18/00: Added INTENDED_DEVICE_FAMILY parameter to altdpram.
// 08/18/00: Fixed underflow/overflow checking problem in dcfifo.
// 08/14/00: Removed DEVICE_FAMILY and OPTIMIZE_FOR_SPEED parameters from
//           scfifo.
// 08/11/00: dcfifo and scfifo are fully implemented.
// 08/02/00: Added altdpram.  scfifo and dcfifo are copied from lpm_fifo
//           and lpm_fifo_dc respectively; however, some Altera-specific
//           parameters are not implemented.
//---------------------------------------------------------------------------

//--------------------------------------------------------------------------
// altdpram megafunction
//--------------------------------------------------------------------------
`timescale 1 ps / 1 ps

module altdpram (wren, data, wraddress, inclock, inclocken, rden, rdaddress,
                 outclock, outclocken, aclr, q);

    parameter width = 1;
    parameter widthad = 1;
    parameter numwords = 0;
    parameter lpm_file = "UNUSED";
    parameter lpm_hint = "USE_EAB=ON";
    parameter indata_reg = "UNREGISTERED";
    parameter indata_aclr = "OFF";
    parameter wraddress_reg = "UNREGISTERED";
    parameter wraddress_aclr = "OFF";
    parameter wrcontrol_reg = "UNREGISTERED";
    parameter wrcontrol_aclr = "OFF";
    parameter rdaddress_reg = "UNREGISTERED";
    parameter rdaddress_aclr = "OFF";
    parameter rdcontrol_reg = "UNREGISTERED";
    parameter rdcontrol_aclr = "OFF";
    parameter outdata_reg = "UNREGISTERED";
    parameter outdata_aclr = "OFF";
    parameter intended_device_family = "APEX20KE";
 
    input  [width-1:0] data;
    input  [widthad-1:0] wraddress, rdaddress;
    input  inclock, inclocken, outclock, outclocken;
    input  wren, rden, aclr;
    output [width-1:0] q;

	// internal reg 
    reg [width-1:0] mem_data [(1<<widthad)-1:0];
    reg [width-1:0] i_data_reg, i_data_tmp, i_q_reg, i_q_tmp;
    reg [widthad-1:0] i_wraddress_reg, i_wraddress_tmp;
    reg [widthad-1:0] i_rdaddress_reg, i_rdaddress_tmp;
    reg i_wren_reg, i_wren_tmp, i_rden_reg, i_rden_tmp;
    reg [8*256:1] ram_initf;
    reg mem_updated;
    integer i, i_numwords;

    tri0 inclock;
    tri1 inclocken;
    tri0 outclock;
    tri1 outclocken;
    tri0 wren;
	tri1 rden;
    tri0 aclr;
			   
    buf (i_inclock, inclock);
    buf (i_inclocken, inclocken);
    buf (i_outclock, outclock);
    buf (i_outclocken, outclocken);
    buf (i_wren, wren);
    buf (i_rden, rden);
    buf (i_aclr, aclr);


	initial
	begin
        // Check for invalid parameters
        if (width <= 0)
            $display("Error! width parameter must be greater than 0.");
        if (widthad <= 0)
            $display("Error! widthad parameter must be greater than 0.");
        if (numwords < 0)
            $display("Error! numwords parameter must be greater than 0.");

        if ((indata_reg !== "INCLOCK") && (indata_reg !== "UNREGISTERED"))
            $display("Error! indata_reg must be INCLOCK or UNREGISTERED.");
        if ((outdata_reg !== "OUTCLOCK") && (outdata_reg !== "UNREGISTERED"))
            $display("Error! outdata_reg must be OUTCLOCK or UNREGISTERED.");
        if ((wraddress_reg !== "INCLOCK") && (wraddress_reg !== "UNREGISTERED"))
            $display("Error! wraddress_reg must be INCLOCK or UNREGISTERED.");
        if ((wrcontrol_reg !== "INCLOCK") && (wrcontrol_reg !== "UNREGISTERED"))
            $display("Error! wrcontrol_reg must be INCLOCK or UNREGISTERED.");
        if ((rdaddress_reg !== "INCLOCK") && (rdaddress_reg !== "OUTCLOCK") && (rdaddress_reg !== "UNREGISTERED"))
            $display("Error! rdaddress_reg must be INCLOCK, OUTCLOCK or UNREGISTERED.");
        if ((rdcontrol_reg !== "INCLOCK") && (rdcontrol_reg !== "OUTCLOCK") && (rdcontrol_reg !== "UNREGISTERED"))
            $display("Error! rdcontrol_reg must be INCLOCK, OUTCLOCK or UNREGISTERED.");
				
        if ((indata_aclr !== "ON") && (indata_aclr !== "OFF"))
            $display("Error! indata_aclr must be ON or OFF.");
        if ((outdata_aclr !== "ON") && (outdata_aclr !== "OFF"))
            $display("Error! outdata_aclr must be ON or OFF.");
        if ((wraddress_aclr !== "ON") && (wraddress_aclr !== "OFF"))
            $display("Error! wraddress_aclr must be ON or OFF.");
        if ((wrcontrol_aclr !== "ON") && (wrcontrol_aclr !== "OFF"))
            $display("Error! wrcontrol_aclr must be ON or OFF.");
        if ((rdaddress_aclr !== "ON") && (rdaddress_aclr !== "OFF"))
            $display("Error! rdaddress_aclr must be ON or OFF.");
        if ((rdcontrol_aclr !== "ON") && (rdcontrol_aclr !== "OFF"))
            $display("Error! rdcontrol_aclr must be ON or OFF.");

        // Initialize mem_data
        i_numwords = (numwords) ? numwords : 1<<widthad;
        if (lpm_file == "UNUSED")
            for (i=0; i<i_numwords; i=i+1)
                mem_data[i] = 0;
        else
		begin
            $convert_hex2ver(lpm_file, width, ram_initf);
			$readmemh(ram_initf, mem_data);
		end

        mem_updated = 0;

        // Initialize registers
        i_data_reg = 0;
        i_data_tmp = 0;
        i_wren_reg = 0;
        i_wren_tmp = 0;
        i_rden_reg = 0;
        i_rden_tmp = 0;
        i_wraddress_reg = 0;
        i_wraddress_tmp = 0;
        i_rdaddress_reg = 0;
        i_rdaddress_tmp = 0;
        i_q_reg = 0;
        i_q_tmp = 0;
	end


    //=========
    // Clocks
    //=========

    always @(negedge i_inclock)
	begin
        if (indata_reg == "INCLOCK")
        begin
            if (i_aclr == 1 && indata_aclr == "ON")
                i_data_reg = 0;
            else if (i_inclocken == 1 && $time > 0)
                i_data_reg = data;
        end

        if (wraddress_reg == "INCLOCK")
        begin
            if (i_aclr == 1 && wrcontrol_aclr == "ON")
                i_wraddress_reg = 0;
            else if (i_inclocken == 1 && $time > 0)
                i_wraddress_reg = wraddress;
        end

        if (wrcontrol_reg == "INCLOCK")
        begin
            if (i_aclr == 1 && wrcontrol_aclr == "ON")
                i_wren_reg = 0;
            else if (i_inclocken == 1 && $time > 0)
                i_wren_reg = i_wren;
        end
    end

    always @(posedge i_inclock)
    begin
        if (rdaddress_reg == "INCLOCK")
        begin
            if (i_aclr == 1 && rdaddress_aclr == "ON")
                i_rdaddress_reg = 0;
            else if (i_inclocken == 1 && $time > 0)
                i_rdaddress_reg = rdaddress;
        end

        if (rdcontrol_reg == "INCLOCK")
        begin
            if (i_aclr == 1 && rdcontrol_aclr == "ON")
                i_rden_reg = 0;
            else if (i_inclocken == 1 && $time > 0)
                i_rden_reg = i_rden;
        end
    end

    always @(posedge i_outclock)
	begin
        if (outdata_reg == "OUTCLOCK")
        begin
            if (i_aclr == 1 && outdata_aclr == "ON")
                i_q_reg = 0;
            else if (i_outclocken == 1 && $time > 0)
                i_q_reg = i_q_tmp;
        end

        if (rdaddress_reg == "OUTCLOCK")
        begin
            if (i_aclr == 1 && rdaddress_aclr == "ON")
                i_rdaddress_reg = 0;
            else if (i_outclocken == 1 && $time > 0)
                i_rdaddress_reg = rdaddress;
        end

        if (rdcontrol_reg == "OUTCLOCK")
        begin
            if (i_aclr == 1 && rdcontrol_aclr == "ON")
                i_rden_reg = 0;
            else if (i_outclocken == 1 && $time > 0)
                i_rden_reg = i_rden;
        end
    end


    //=========
    // Memory
    //=========

    always @(i_data_tmp or i_wren_tmp or i_wraddress_tmp)
	begin
        if (i_wren_tmp == 1)
        begin
            mem_data[i_wraddress_tmp] = i_data_tmp;
            mem_updated = ~mem_updated;
        end
	end

    always @(i_rden_tmp or i_rdaddress_tmp or mem_updated)
	begin
        if (i_rden_tmp == 1)
            i_q_tmp = mem_data[i_rdaddress_tmp];
	end


    //=======
    // Sync
    //=======

    always @(wraddress or i_wraddress_reg or i_aclr)
        i_wraddress_tmp = (i_aclr == 1 && wraddress_aclr == "ON") ?
                          0 :
                              ((wrcontrol_reg == "INCLOCK") ?
                               i_wraddress_reg : wraddress);
    always @(rdaddress or i_rdaddress_reg or i_aclr)
        i_rdaddress_tmp = (i_aclr == 1 && rdaddress_aclr == "ON") ?
                          0 :
                              ((rdaddress_reg == "INCLOCK" || rdaddress_reg == "OUTCLOCK") ?
                               i_rdaddress_reg : rdaddress);
    always @(i_wren or i_wren_reg or i_aclr)
        i_wren_tmp = (i_aclr == 1 && wrcontrol_aclr == "ON") ?
                     0 :
                         ((wrcontrol_reg == "INCLOCK") ?
                          i_wren_reg : i_wren);
    always @(i_rden or i_rden_reg or i_aclr)
        i_rden_tmp = (i_aclr == 1 && rdcontrol_aclr == "ON") ?
                     0 :
                         ((rdcontrol_reg == "INCLOCK" || rdcontrol_reg == "OUTCLOCK") ?
                          i_rden_reg : i_rden);
    always @(data or i_data_reg or i_aclr)
        i_data_tmp = (i_aclr == 1 && indata_aclr == "ON") ?
                     0 :
                         ((indata_reg == "INCLOCK") ?
                          i_data_reg : data);
    assign q = (i_aclr == 1 && outdata_aclr == "ON") ?
               0 :
                   ((outdata_reg == "OUTCLOCK") ?
                    i_q_reg : i_q_tmp);

endmodule // altdpram


//--------------------------------------------------------------------------
// scfifo megafunction
//--------------------------------------------------------------------------

module scfifo ( data, clock, wrreq, rdreq, aclr, sclr,
                q, usedw, full, empty, almost_full, almost_empty );

	parameter lpm_width  = 1;
	parameter lpm_widthu  = 1;
	parameter lpm_numwords = 2;
	parameter lpm_showahead = "OFF";
    parameter lpm_hint = "USE_EAB=ON";
    parameter underflow_checking = "ON";
    parameter overflow_checking = "ON";
    parameter allow_rwcycle_when_full = "OFF";
    parameter almost_full_value = 1;
    parameter almost_empty_value = 0;
    // parameters that should be stored in lpm_hint
    parameter use_eab = "ON";
    //parameter maximize_speed = 5;
    //parameter device_family = "";
    //parameter optimize_for_speed = 5;

    input  [lpm_width-1:0] data;
    input  clock;
    input  wrreq;
    input  rdreq;
    input  aclr;
    input  sclr;
	output [lpm_width-1:0] q;
	output [lpm_widthu-1:0] usedw;
	output full;
	output empty;
    output almost_full;
    output almost_empty;


	// internal reg
	reg [lpm_width-1:0] mem_data [lpm_numwords-1:0];
	reg [lpm_width-1:0] tmp_q;
	reg [lpm_widthu+1:0] count_id;
	reg [lpm_widthu-1:0] write_id;
	reg [lpm_widthu-1:0] read_id;
    reg empty_flag, full_flag;
    reg valid_rreq, valid_wreq;
	integer i;

	tri0 aclr;
	tri0 sclr;

	buf (i_aclr, aclr);
	buf (i_sclr, sclr);


	initial
	begin
  
		if (lpm_width <= 0)
			$display("Error! lpm_width must be greater than 0.");
		if (lpm_numwords <= 1)
			$display("Error! lpm_numwords must be greater than or equal to 2.");
		if ((lpm_widthu !=1) && (lpm_numwords > (1 << lpm_widthu)))
			$display("Error! lpm_numwords MUST equal to the ceiling of log2(lpm_widthu).");
		if (lpm_numwords <= (1 << (lpm_widthu-1)))
			$display("Error! lpm_widthu is too big for the specified lpm_numwords.");

		for (i = 0; i < lpm_numwords; i=i+1)
            mem_data[i] = 0;

		full_flag = 0;
		empty_flag = 1;
		read_id = 0;
		write_id = 0;
		count_id = 0;
        tmp_q = 0;
	end

    always @(i_aclr)
    begin
		if (i_aclr)
		begin
            tmp_q = 0;
			full_flag = 0;
			empty_flag = 1;
			read_id = 0;
			write_id = 0;
			count_id = 0;
			if (lpm_showahead == "ON")
				tmp_q = mem_data[0];
		end
    end

    always @(posedge clock)
	begin
        if (!i_aclr)
		begin
			if (i_sclr)
			begin
				tmp_q = mem_data[read_id];
				full_flag = 0;
				empty_flag = 1;
				read_id = 0;
				write_id = 0;
				count_id = 0;
				if (lpm_showahead == "ON")
					tmp_q = mem_data[0];
			end
			else
			begin
                valid_rreq = (underflow_checking == "OFF")
                               ? rdreq
                               : rdreq && !empty_flag;
                valid_wreq = (overflow_checking == "OFF")
                               ? wrreq
                               : (allow_rwcycle_when_full == "ON")
                                   ? wrreq && (!full_flag || rdreq)
                                   : wrreq && !full_flag;

				// both WRITE and READ
                if (valid_wreq && valid_rreq)
				begin
					mem_data[write_id] = data;
					if (write_id >= lpm_numwords-1)
						write_id = 0;
					else
						write_id = write_id + 1;

					tmp_q = mem_data[read_id];
					if (read_id >= lpm_numwords-1)
						read_id = 0;
					else
						read_id = read_id + 1;
					if (lpm_showahead == "ON")
						tmp_q = mem_data[read_id];
				end

				// WRITE
                else if (valid_wreq)
				begin
					mem_data[write_id] = data;
					if (lpm_showahead == "ON")
						tmp_q = mem_data[read_id];
					count_id = count_id + 1;
					empty_flag = 0;
					if (count_id >= lpm_numwords)
					begin
						full_flag = 1;
						count_id = lpm_numwords;
					end
					if (write_id >= lpm_numwords-1)
						write_id = 0;
					else
						write_id = write_id + 1;
				end
							
				// READ
                else if (valid_rreq)
				begin
					tmp_q = mem_data[read_id];
					count_id = count_id - 1;
					full_flag = 0;
					if (count_id <= 0)
					begin
						empty_flag = 1;
						count_id = 0;
					end
					if (read_id >= lpm_numwords-1)
						read_id = 0;
					else
    					read_id = read_id + 1;
					if (lpm_showahead == "ON")
						tmp_q = mem_data[read_id];
				end
			end
		end
	end

	assign q = tmp_q;
	assign full = full_flag;
	assign empty = empty_flag;
	assign usedw = count_id;
    assign almost_full = (count_id >= almost_full_value) ? 1 : 0;
    assign almost_empty = (count_id < almost_empty_value) ? 1 : 0;

endmodule // scfifo


//--------------------------------------------------------------------------
// dcfifo megafunction
//--------------------------------------------------------------------------

module dcfifo_dffpipe ( d, q, clock, aclr );

    parameter lpm_delay = 1;
    parameter lpm_width = 64;

    input [lpm_width-1:0] d;
    input clock;
    input aclr;
    output [lpm_width-1:0] q;    

	// internal reg
    reg [lpm_width-1:0] dffpipe [lpm_delay:0];
    reg [lpm_width-1:0] i_q;
    integer delay, i;

	tri0 aclr;
	buf (i_aclr, aclr);


    initial
    begin
        delay = lpm_delay-1;
        for (i=0; i<lpm_delay; i=i+1)
            dffpipe[i] = 0;
        i_q = 0;
    end

    always @(d)
    begin
        if (lpm_delay == 0 && !i_aclr)
            i_q = (i_aclr) ? 0 : d;
    end

    always @(i_aclr)
    begin
        if (i_aclr)
        begin
            for (i=0; i<lpm_delay; i=i+1)
                dffpipe[i] = 0;
            i_q = 0;
        end
    end

    always @(posedge clock)
    begin
        if (!i_aclr && lpm_delay > 0 && $time > 0)
        begin
            if (delay > 0)
                for (i=delay; i>0; i=i-1)
                    dffpipe[i] = dffpipe[i-1];
            dffpipe[0] = d;
            i_q = dffpipe[delay];
        end
    end

    assign q = i_q;

endmodule // dcfifo_dffpipe

//-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- ---

module dcfifo_fefifo ( usedw_in, wreq, rreq, empty, full, clock, aclr );

    parameter lpm_widthad = 1;
    parameter lpm_numwords = 1;
    parameter underflow_checking = "ON";
    parameter overflow_checking = "ON";
    parameter lpm_mode = "READ";

    input [lpm_widthad-1:0] usedw_in;
    input wreq, rreq;
    input empty, full;
    input clock;
    input aclr;

	// internal reg
    reg [1:0] sm_empty;
    reg lrreq, valid_rreq;
    reg i_empty, i_full;
    integer almostfull;

	tri0 aclr;
	buf (i_aclr, aclr);


    initial
    begin
        if(lpm_mode != "READ" && lpm_mode != "WRITE")
            $display("Error! lpm_mode must be READ or WRITE.");
        if(underflow_checking != "ON" && underflow_checking != "OFF")
            $display("Error! underflow_checking must be ON or OFF.");
        if(overflow_checking != "ON" && overflow_checking != "OFF")
            $display("Error! overflow_checking must be ON or OFF.");

        sm_empty = 2'b00;
        lrreq = 1'b0;
        i_empty = 1'b1;
        i_full = 1'b0;
        almostfull = (lpm_numwords >= 3) ? lpm_numwords-3 : 0;
    end

    always @(rreq or i_empty)
        valid_rreq = (underflow_checking == "OFF") ? rreq
                                                   : rreq && !i_empty;
    always @(i_aclr)
    begin
        if (i_aclr)
        begin
            sm_empty = 2'b00;
            lrreq = 1'b0;
            i_empty = 1'b1;
            i_full = 1'b0;
        end
    end

    always @(posedge clock)
    begin
        if (!i_aclr && $time > 0)
        begin
            if (lpm_mode == "READ")
            begin
                casex (sm_empty)
                    2'b00:                          // state_empty
                        if (usedw_in != 0)
                            sm_empty = 2'b01;
                    2'b01:                          // state_non_empty
                        if (rreq && ((usedw_in==1 && !lrreq) || (usedw_in==2 && lrreq)))
                            sm_empty = 2'b10;
                    2'b10:                          // state_emptywait
                        sm_empty = (usedw_in > 1) ? 2'b01 : 2'b00;
                endcase
                if ($time > 0)
                    lrreq = valid_rreq;
            end
            else if (lpm_mode == "WRITE")
            begin
                casex (sm_empty)
                    2'b00:                          // state_empty
                        if (wreq)
                            sm_empty = 2'b01;
                    2'b01:                          // state_one
                        if (!wreq)
                            sm_empty = 2'b11;
                    2'b11:                          // state_non_empty
                        if (wreq)
                            sm_empty = 2'b01;
                        else if (usedw_in == 0)
                            sm_empty = 2'b00;
                endcase
            end
        end
        i_empty = !sm_empty[0];
        i_full = (!i_aclr && $time>0 && usedw_in>=almostfull) ? 1'b1 : 1'b0;
    end

    assign empty = i_empty;
    assign full = i_full;

endmodule // dcfifo_fefifo

//-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- ---

module dcfifo_async ( data, rdclk, wrclk, aclr, rdreq, wrreq, rdfull,
                      wrfull, rdempty, wrempty, rdusedw, wrusedw, q );

	parameter lpm_width = 1;
	parameter lpm_widthu = 1;
	parameter lpm_numwords = 2;
	parameter lpm_showahead = "OFF";
    parameter underflow_checking = "ON";
    parameter overflow_checking = "ON";
    parameter delay_rdusedw = 1;
    parameter delay_wrusedw = 1;
    parameter rdsync_delaypipe = 3;
    parameter wrsync_delaypipe = 3;
    parameter use_eab = "ON";

	input [lpm_width-1:0] data;
    input rdclk;
    input wrclk;
	input wrreq;
	input rdreq;
	input aclr;
	output rdfull;
	output wrfull;
	output rdempty;
	output wrempty;
	output [lpm_widthu-1:0] rdusedw;
	output [lpm_widthu-1:0] wrusedw;
	output [lpm_width-1:0] q;

	// internal reg
    reg [lpm_width-1:0] mem_data [(1<<lpm_widthu)-1:0];
    reg [lpm_widthu-1:0] i_rdptr, i_wrptr;
    wire [lpm_widthu-1:0] w_rdptrrg, w_wrdelaycycle;
    wire [lpm_widthu-1:0] w_ws_nbrp, w_rs_nbwp, w_ws_dbrp, w_rs_dbwp;
    wire [lpm_widthu-1:0] w_wr_dbuw, w_rd_dbuw, w_rdusedw, w_wrusedw;
    reg [lpm_widthu-1:0] i_wr_udwn, i_rd_udwn;
    wire w_rdempty, w_wrempty, wrdfull, w_wrfull;
    reg i_rdempty, i_wrempty;
    reg i_rdfull, i_wrfull;
    reg i_rden, i_wren, i_rdenclock;
    reg [lpm_width-1:0] i_q;
    integer i;

    reg [lpm_width-1:0] i_data_tmp, i_data_reg;
    reg [lpm_width-1:0] i_q_tmp, i_q_reg;
    reg [lpm_widthu-1:0] i_rdptr_tmp, i_wrptr_tmp, i_wrptr_reg;
    reg i_wren_tmp, i_wren_reg;

	tri0 aclr;
	buf (i_aclr, aclr);


    initial
    begin
        if(lpm_showahead != "ON" && lpm_showahead != "OFF")
            $display("Error! lpm_showahead must be ON or OFF.");
        if(underflow_checking != "ON" && underflow_checking != "OFF")
            $display("Error! underflow_checking must be ON or OFF.");
        if(overflow_checking != "ON" && overflow_checking != "OFF")
            $display("Error! overflow_checking must be ON or OFF.");
        if(use_eab != "ON" && use_eab != "OFF")
            $display("Error! use_eab must be ON or OFF.");

        i_rdptr = 0;
        i_wrptr = 0;
        i_rdempty = 1;
        i_wrempty = 1;
        i_rdfull = 0;
        i_wrfull = 0;
        i_rden = 0;
        i_wren = 0;
        i_q = 0;

        i_q_tmp = 0;
        i_q_reg = 0;
        i_rdptr_tmp = 0;
        i_wrptr_tmp = 0;
        i_wrptr_reg = 0;
        i_wren_tmp = 0;
        i_wren_reg = 0;

        for (i=0; i<(1<<lpm_widthu); i=i+1)
            mem_data[i] = 0;
    end


    //--------
    // FIFOram
    //--------

    always @(rdreq or i_rdempty)
        i_rden = (underflow_checking == "OFF") ? rdreq
                                               : rdreq && !i_rdempty;
    always @(wrreq or i_wrfull)
        i_wren = (overflow_checking == "OFF") ? wrreq
                                              : wrreq && !i_wrfull;

    // FIFOram_sync
    always @(i_data_reg or i_q_tmp or i_q_reg or i_aclr or
             i_rdptr or i_wren_reg or i_wrptr_reg)
    begin
        if (i_aclr)
        begin
            i_wrptr_tmp = 0;
            i_rdptr_tmp = 0;
            i_wren_tmp = 0;
            i_data_tmp = 0;
            i_q = (lpm_showahead == "ON") ? i_q_tmp : 0;
        end
        else
        begin
            i_wrptr_tmp = i_wrptr_reg;
            i_rdptr_tmp = i_rdptr;
            i_wren_tmp = i_wren_reg;
            i_data_tmp = i_data_reg;
            i_q = (lpm_showahead == "ON") ? i_q_tmp : i_q_reg;
        end
    end

    // FIFOram_aclr
    always @(i_aclr)
    begin
        if (i_aclr)
        begin
            i_data_reg = 0;
            i_wrptr_reg = 0;
            i_wren_reg = 0;
            i_q_reg = 0;
        end
    end

    // FIFOram_wrclk
    always @(posedge wrclk)
    begin
        if (!i_aclr && $time > 0)
        begin
            i_data_reg = data;
            i_wrptr_reg = i_wrptr;
            i_wren_reg = i_wren;
        end
    end

    // FIFOram_rdclk
    always @(posedge rdclk)
    begin
        if (!i_aclr && i_rden && $time > 0)
            i_q_reg = i_q_tmp;
    end

    // FIFOram_memory_read
    always @(i_rdptr_tmp)
        i_q_tmp = mem_data[i_rdptr_tmp];

    // FIFOram_memory_write_eab
    always @(negedge wrclk)
    begin
        if (use_eab == "ON")
        begin
            if (i_wren_tmp)
                mem_data[i_wrptr_tmp] = i_data_tmp;
            i_q_tmp = mem_data[i_rdptr_tmp];
        end
    end

    // FIFOram_memory_write_lcell
    always @(posedge wrclk)
    begin
        if (use_eab == "OFF")
        begin
            if (i_wren_tmp)
                mem_data[i_wrptr_tmp] = i_data_tmp;
            i_q_tmp = mem_data[i_rdptr_tmp];
        end
    end


    //---------
    // Counters
    //---------

    // rdptr
    always @(i_aclr)
        if (i_aclr)
            i_rdptr = 0;
    always @(posedge rdclk)
        if (!i_aclr && i_rden && $time > 0)
            i_rdptr = ((i_rdptr < (1<<lpm_widthu)-1) || (underflow_checking == "OFF"))
                        ? i_rdptr+1 : 0;

    // wrptr
    always @(i_aclr)
        if (i_aclr)
            i_wrptr = 0;
    always @(posedge wrclk)
        if (!i_aclr && i_wren && $time > 0)
            i_wrptr = ((i_wrptr < (1<<lpm_widthu)-1) || (overflow_checking == "OFF"))
                        ? i_wrptr+1 : 0;

    //-------------------
    // Delays & DFF Pipes
    //-------------------

    always @(negedge rdclk)  i_rdenclock = 0;
    always @(posedge rdclk)  if (i_rden) i_rdenclock = 1;

    dcfifo_dffpipe RDPTR_D ( .d (i_rdptr), .q (w_rdptrrg),
                             .clock (i_rdenclock), .aclr (i_aclr) );
    dcfifo_dffpipe WRPTR_D ( .d (i_wrptr), .q (w_wrdelaycycle),
                             .clock (wrclk), .aclr (i_aclr) );
    defparam
        RDPTR_D.lpm_delay = 0,
        RDPTR_D.lpm_width = lpm_widthu,
        WRPTR_D.lpm_delay = 1,
        WRPTR_D.lpm_width = lpm_widthu;

    dcfifo_dffpipe WS_NBRP ( .d (w_rdptrrg), .q (w_ws_nbrp),
                             .clock (wrclk), .aclr (i_aclr) );
    dcfifo_dffpipe RS_NBWP ( .d (w_wrdelaycycle), .q (w_rs_nbwp),
                             .clock (rdclk), .aclr (i_aclr) );
    dcfifo_dffpipe WS_DBRP ( .d (w_ws_nbrp), .q (w_ws_dbrp),
                             .clock (wrclk), .aclr (i_aclr) );
    dcfifo_dffpipe RS_DBWP ( .d (w_rs_nbwp), .q (w_rs_dbwp),
                             .clock (rdclk), .aclr (i_aclr) );   
    defparam
        WS_NBRP.lpm_delay = wrsync_delaypipe,
        WS_NBRP.lpm_width = lpm_widthu,
        RS_NBWP.lpm_delay = rdsync_delaypipe,
        RS_NBWP.lpm_width = lpm_widthu,
        WS_DBRP.lpm_delay = 1,              // gray_delaypipe
        WS_DBRP.lpm_width = lpm_widthu,
        RS_DBWP.lpm_delay = 1,              // gray_delaypipe
        RS_DBWP.lpm_width = lpm_widthu;

    always @(i_wrptr or w_ws_dbrp)
        i_wr_udwn = i_wrptr - w_ws_dbrp;

    always @(i_rdptr or w_rs_dbwp)
        i_rd_udwn = w_rs_dbwp - i_rdptr;

    dcfifo_dffpipe WRUSEDW ( .d (i_wr_udwn), .q (w_wrusedw),
                             .clock (wrclk), .aclr (i_aclr) );
    dcfifo_dffpipe RDUSEDW ( .d (i_rd_udwn), .q (w_rdusedw),
                             .clock (rdclk), .aclr (i_aclr) );
    dcfifo_dffpipe WR_DBUW ( .d (i_wr_udwn), .q (w_wr_dbuw),
                             .clock (wrclk), .aclr (i_aclr) );
    dcfifo_dffpipe RD_DBUW ( .d (i_rd_udwn), .q (w_rd_dbuw),
                             .clock (rdclk), .aclr (i_aclr) );
    defparam
        WRUSEDW.lpm_delay = delay_wrusedw,
        WRUSEDW.lpm_width = lpm_widthu,
        RDUSEDW.lpm_delay = delay_rdusedw,
        RDUSEDW.lpm_width = lpm_widthu,
        WR_DBUW.lpm_delay = 1,              // wrusedw_delaypipe
        WR_DBUW.lpm_width = lpm_widthu,
        RD_DBUW.lpm_delay = 1,              // rdusedw_delaypipe
        RD_DBUW.lpm_width = lpm_widthu;


    //-----------
    // Full/Empty
    //-----------

    dcfifo_fefifo WR_FE ( .usedw_in (w_wr_dbuw), .wreq (wrreq), .rreq (rdreq),
                          .clock (wrclk), .aclr (i_aclr),
                          .empty (w_wrempty), .full (w_wrfull) );
    dcfifo_fefifo RD_FE ( .usedw_in (w_rd_dbuw), .rreq (rdreq), .wreq(wrreq),
                          .clock (rdclk), .aclr (i_aclr),
                          .empty (w_rdempty), .full (w_rdfull) );

    defparam
        WR_FE.lpm_widthad = lpm_widthu,
        WR_FE.lpm_numwords = lpm_numwords,
        WR_FE.underflow_checking = underflow_checking,
        WR_FE.overflow_checking = overflow_checking,
        WR_FE.lpm_mode = "WRITE",
        RD_FE.lpm_widthad = lpm_widthu,
        RD_FE.lpm_numwords = lpm_numwords,
        RD_FE.underflow_checking = underflow_checking,
        RD_FE.overflow_checking = overflow_checking,
        RD_FE.lpm_mode = "READ";

    always @(w_wrfull)  i_wrfull = w_wrfull;
    always @(w_rdfull)  i_rdfull = w_rdfull;
    always @(w_wrempty)  i_wrempty = w_wrempty;
    always @(w_rdempty)  i_rdempty = w_rdempty;


    //--------
    // Outputs
    //--------

    assign q = i_q;
    assign wrfull = i_wrfull;
    assign rdfull = i_rdfull;
    assign wrempty = i_wrempty;
    assign rdempty = i_rdempty;
    assign wrusedw = w_wrusedw;
    assign rdusedw = w_rdusedw;

endmodule // dcfifo_async

//-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- ---

module dcfifo_sync ( data, rdclk, wrclk, aclr, rdreq, wrreq, rdfull,
                     wrfull, rdempty, wrempty, rdusedw, wrusedw, q );

	parameter lpm_width = 1;
	parameter lpm_widthu = 1;
	parameter lpm_numwords = 2;
	parameter lpm_showahead = "OFF";
    parameter underflow_checking = "ON";
    parameter overflow_checking = "ON";
    parameter use_eab = "ON";

	input [lpm_width-1:0] data;
    input rdclk;
    input wrclk;
	input wrreq;
	input rdreq;
	input aclr;
	output rdfull;
	output wrfull;
	output rdempty;
	output wrempty;
	output [lpm_widthu-1:0] rdusedw;
	output [lpm_widthu-1:0] wrusedw;
	output [lpm_width-1:0] q;


	// internal reg
    reg [lpm_width-1:0] mem_data [(1<<lpm_widthu)-1:0];
    reg [lpm_widthu:0] i_rdptr, i_wrptr;
    reg [lpm_widthu:0] i_rdusedw, i_wrusedw;
    reg i_rdempty, i_wrempty;
    reg i_rdfull, i_wrfull;
    reg i_rden, i_wren;
    reg [lpm_width-1:0] i_q;
    wire [lpm_widthu:0] w_rdptr_s, w_wrptr_s;
    wire [lpm_widthu:0] w_wrptr_r;
    integer cnt_mod, i;

    reg [lpm_width-1:0] i_data_tmp, i_data_reg;
    reg [lpm_width-1:0] i_q_tmp, i_q_reg;
    reg [lpm_widthu-1:0] i_rdptr_tmp, i_wrptr_tmp, i_wrptr_reg;
    reg i_wren_tmp, i_wren_reg;

	tri0 aclr;
	buf (i_aclr, aclr);


    initial
    begin
        if(lpm_showahead != "ON" && lpm_showahead != "OFF")
            $display("Error! lpm_showahead must be ON or OFF.");
        if(underflow_checking != "ON" && underflow_checking != "OFF")
            $display("Error! underflow_checking must be ON or OFF.");
        if(overflow_checking != "ON" && overflow_checking != "OFF")
            $display("Error! overflow_checking must be ON or OFF.");
        if(use_eab != "ON" && use_eab != "OFF")
            $display("Error! use_eab must be ON or OFF.");
        if(lpm_numwords>(1<<lpm_widthu))
            $display("Error! lpm_numwords must be less than or equal to 2**lpm_widthu.");

        i_rdptr = 0;
        i_wrptr = 0;
        i_rdempty = 1;
        i_wrempty = 1;
        i_rdfull = 0;
        i_wrfull = 0;
        i_rden = 0;
        i_wren = 0;
        i_q = 0;

        i_q_tmp = 0;
        i_q_reg = 0;
        i_rdptr_tmp = 0;
        i_wrptr_tmp = 0;
        i_wrptr_reg = 0;

        i_wren_tmp = 0;
        i_wren_reg = 0;

        cnt_mod = (lpm_numwords == 1<<lpm_widthu) ? 1<<(lpm_widthu+1)
                                                  : 1<<lpm_widthu;

        for (i=0; i<(1<<lpm_widthu); i=i+1)
            mem_data[i] = 0;
    end


    //--------
    // FIFOram
    //--------

    always @(rdreq or i_rdempty)
        i_rden = (underflow_checking == "OFF") ? rdreq
                                               : rdreq && !i_rdempty;
    always @(wrreq or i_wrfull)
        i_wren = (overflow_checking == "OFF") ? wrreq
                                              : wrreq && !i_wrfull;

    // FIFOram_sync
    always @(i_data_reg or i_q_tmp or i_q_reg or i_aclr or
             i_rdptr or i_wren_reg or i_wrptr_reg)
        if (i_aclr)
        begin
            i_wrptr_tmp = 0;
            i_rdptr_tmp = 0;
            i_wren_tmp = 0;
            i_data_tmp = 0;
            i_q = (lpm_showahead == "ON") ? i_q_tmp : 0;
        end
        else
        begin
            i_wrptr_tmp = i_wrptr_reg;
            i_rdptr_tmp = i_rdptr[lpm_widthu-1:0];
            i_wren_tmp = i_wren_reg;
            i_data_tmp = i_data_reg;
            i_q = (lpm_showahead == "ON") ? i_q_tmp : i_q_reg;
        end

    // FIFOram_aclr
    always @(i_aclr)
        if (i_aclr)
        begin
            i_data_reg = 0;
            i_wrptr_reg = 0;
            i_wren_reg = 0;
            i_q_reg = 0;
        end

    // FIFOram_wrclk
    always @(posedge wrclk)
        if (!i_aclr && $time > 0)
        begin
            i_data_reg = data;
            i_wrptr_reg = i_wrptr[lpm_widthu-1:0];
            i_wren_reg = i_wren;
        end

    // FIFOram_rdclk
    always @(posedge rdclk)
        if (!i_aclr && i_rden && $time > 0)
            i_q_reg = i_q_tmp;

    // FIFOram_memory_read
    always @(i_rdptr_tmp)
        i_q_tmp = mem_data[i_rdptr_tmp];

    // FIFOram_memory_write
    always @(wrclk)
        if ((use_eab=="ON" ^ wrclk) && $time > 0)
        begin
            if (i_wren_tmp)
                mem_data[i_wrptr_tmp] = i_data_tmp;
            i_q_tmp = mem_data[i_rdptr_tmp];
        end


    //---------
    // Counters
    //---------

    // rdptr
    always @(i_aclr)
        if (i_aclr)
            i_rdptr = 0;
    always @(posedge rdclk)
        if (!i_aclr && i_rden && $time > 0)
            i_rdptr = (i_rdptr < cnt_mod-1) ? i_rdptr+1 : 0;

    // wrptr
    always @(i_aclr)
        if (i_aclr)
            i_wrptr = 0;
    always @(posedge wrclk)
        if (!i_aclr && i_wren && $time > 0)
            i_wrptr = (i_wrptr < cnt_mod-1) ? i_wrptr+1 : 0;


    //--------
    // Delays
    //--------

    dcfifo_dffpipe RDPTR_D ( .d (i_rdptr), .q (w_rdptr_s),
                             .clock (wrclk), .aclr (i_aclr) );
    dcfifo_dffpipe WRPTR_D ( .d (i_wrptr), .q (w_wrptr_r),
                             .clock (wrclk), .aclr (i_aclr) );
    dcfifo_dffpipe WRPTR_E ( .d (w_wrptr_r), .q (w_wrptr_s),
                             .clock (rdclk), .aclr (i_aclr) );
    defparam
        RDPTR_D.lpm_delay = 1,
        RDPTR_D.lpm_width = lpm_widthu+1,
        WRPTR_D.lpm_delay = 1,
        WRPTR_D.lpm_width = lpm_widthu+1,
        WRPTR_E.lpm_delay = 1,
        WRPTR_E.lpm_width = lpm_widthu+1;


    //-----------------
    // Usedw/Full/Empty
    //-----------------

    always @(i_wrptr or w_rdptr_s)
        i_wrusedw = (i_wrptr >= w_rdptr_s) ? i_wrptr - w_rdptr_s
                                           : i_wrptr + cnt_mod - w_rdptr_s;

    always @(i_rdptr or w_wrptr_s)
        i_rdusedw = (w_wrptr_s >= i_rdptr) ? w_wrptr_s - i_rdptr
                                           : w_wrptr_s + cnt_mod - i_rdptr;

    always @(i_wrusedw)
    begin
        i_wrfull = (lpm_numwords==(1<<lpm_widthu) && i_wrusedw[lpm_widthu]) ||
                   (lpm_numwords<(1<<lpm_widthu) && i_wrusedw==lpm_numwords)
                     ? 1 : 0;
        i_wrempty = (i_wrusedw == 0) ? 1 : 0;
    end

    always @(i_rdusedw)
    begin
        i_rdfull = (lpm_numwords==(1<<lpm_widthu) && i_rdusedw[lpm_widthu]) ||
                   (lpm_numwords<(1<<lpm_widthu) && i_rdusedw==lpm_numwords)
                     ? 1 : 0;
        i_rdempty = (i_rdusedw == 0) ? 1 : 0;
    end


    //--------
    // Outputs
    //--------

    assign q = i_q;
    assign wrfull = i_wrfull;
    assign rdfull = i_rdfull;
    assign wrempty = i_wrempty;
    assign rdempty = i_rdempty;
    assign wrusedw = i_wrusedw[lpm_widthu-1:0];
    assign rdusedw = i_rdusedw[lpm_widthu-1:0];

endmodule // dcfifo_sync

//-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- ---

module dcfifo ( data, rdclk, wrclk, aclr, rdreq, wrreq, rdfull,
                wrfull, rdempty, wrempty, rdusedw, wrusedw, q );

	parameter lpm_width = 1;
	parameter lpm_widthu = 1;
	parameter lpm_numwords = 2;
	parameter lpm_showahead = "OFF";
    parameter lpm_hint = "USE_EAB=ON";
    parameter overflow_checking = "ON";
    parameter underflow_checking = "ON";
    parameter delay_rdusedw = 1;
    parameter delay_wrusedw = 1;
    parameter rdsync_delaypipe = 3;
    parameter wrsync_delaypipe = 3;
    // parameters that should be stored in lpm_hint
    parameter use_eab = "ON";
    parameter clocks_are_synchronized = "FALSE";

	input [lpm_width-1:0] data;
    input rdclk;
    input wrclk;
	input wrreq;
	input rdreq;
	input aclr;
	output rdfull;
	output wrfull;
	output rdempty;
	output wrempty;
	output [lpm_widthu-1:0] rdusedw;
	output [lpm_widthu-1:0] wrusedw;
	output [lpm_width-1:0] q;

    wire w_rdfull_s, w_wrfull_s, w_rdempty_s, w_wrempty_s;
    wire w_rdfull_a, w_wrfull_a, w_rdempty_a, w_wrempty_a;
    wire [lpm_widthu-1:0] w_rdusedw_s, w_wrusedw_s;
    wire [lpm_widthu-1:0] w_rdusedw_a, w_wrusedw_a;
    wire [lpm_width-1:0] w_q_s;
    wire [lpm_width-1:0] w_q_a;

    reg i_rdfull_s, i_wrfull_s, i_rdempty_s, i_wrempty_s;
    reg i_rdfull_a, i_wrfull_a, i_rdempty_a, i_wrempty_a;
    reg [lpm_widthu-1:0] i_rdusedw_s, i_wrusedw_s;
    reg [lpm_widthu-1:0] i_rdusedw_a, i_wrusedw_a;
    reg [lpm_width-1:0] i_q_s;
    reg [lpm_width-1:0] i_q_a;

    dcfifo_sync  SYNC  ( .data (data), .rdclk (rdclk), .wrclk (wrclk),
                         .aclr (aclr), .rdreq (rdreq), .wrreq (wrreq),
                         .rdfull (w_rdfull_s), .wrfull (w_wrfull_s),
                         .rdempty (w_rdempty_s), .wrempty (w_wrempty_s),
                         .rdusedw (w_rdusedw_s), .wrusedw (w_wrusedw_s),
                         .q (w_q_s) );

    defparam
        SYNC.lpm_width = lpm_width,
        SYNC.lpm_widthu = lpm_widthu,
        SYNC.lpm_numwords = lpm_numwords,
        SYNC.lpm_showahead = lpm_showahead,
        SYNC.underflow_checking = underflow_checking,
        SYNC.overflow_checking = overflow_checking,
        SYNC.use_eab = use_eab;

    dcfifo_async ASYNC ( .data (data), .rdclk (rdclk), .wrclk (wrclk),
                         .aclr (aclr), .rdreq (rdreq), .wrreq (wrreq),
                         .rdfull (w_rdfull_a), .wrfull (w_wrfull_a),
                         .rdempty (w_rdempty_a), .wrempty (w_wrempty_a),
                         .rdusedw (w_rdusedw_a), .wrusedw (w_wrusedw_a),
                         .q (w_q_a) );

    defparam
        ASYNC.lpm_width = lpm_width,
        ASYNC.lpm_widthu = lpm_widthu,
        ASYNC.lpm_numwords = lpm_numwords,
        ASYNC.lpm_showahead = lpm_showahead,
        ASYNC.underflow_checking = underflow_checking,
        ASYNC.overflow_checking = overflow_checking,
        ASYNC.delay_rdusedw = delay_rdusedw,
        ASYNC.delay_wrusedw = delay_wrusedw,
        ASYNC.rdsync_delaypipe = rdsync_delaypipe,
        ASYNC.wrsync_delaypipe = wrsync_delaypipe,
        ASYNC.use_eab = use_eab;

    assign rdfull = (clocks_are_synchronized == "FALSE") ? w_rdfull_a : w_rdfull_s;
    assign wrfull = (clocks_are_synchronized == "FALSE") ? w_wrfull_a : w_wrfull_s;
    assign rdempty = (clocks_are_synchronized == "FALSE") ? w_rdempty_a : w_rdempty_s;
    assign wrempty = (clocks_are_synchronized == "FALSE") ? w_wrempty_a : w_wrempty_s;
    assign rdusedw = (clocks_are_synchronized == "FALSE") ? w_rdusedw_a : w_rdusedw_s;
    assign wrusedw = (clocks_are_synchronized == "FALSE") ? w_wrusedw_a : w_wrusedw_s;
    assign q = (clocks_are_synchronized == "FALSE") ? w_q_a : w_q_s;

endmodule // dcfifo
