--
-- Copyright (C) 1988-1999 Altera Corporation
--
-- Any megafunction design, and related net list (encrypted or decrypted),
-- support information, device programming or simulation file, and any
-- other associated documentation or information provided by Altera or a
-- partner under Altera's Megafunction Partnership Program may be used only
-- to program PLD devices (but not masked PLD devices) from Altera.  Any
-- other use of such megafunction design, net list, support information,
-- device programming or simulation file, or any other related
-- documentation or information is prohibited for any other purpose,
-- including, but not limited to modification, reverse engineering, de-
-- compiling, or use with any other silicon devices, unless such use is
-- explicitly licensed under a separate agreement with Altera or a
-- megafunction partner.  Title to the intellectual property, including
-- patents, copyrights, trademarks, trade secrets, or maskworks, embodied
-- in any such megafunction design, net list, support information, device
-- programming or simulation file, or any other related documentation or
-- information provided by Altera or a megafunction partner, remains with
-- Altera, the megafunction partner, or their respective licensors.  No
-- other licenses, including any licenses needed under any third party's
-- intellectual property, are provided herein.
----------------------------------------------------------------------------
----------------------------------------------------------------------------
-- Version 1.01 Date: 08/23/00 for MAX+plus II
--
-- 08/22/00: Added synchronous mode to dcfifo.
-- 08/18/00: Added INTENDED_DEVICE_FAMILY parameter to altdpram.
-- 08/18/00: Fixed underflow/overflow checking problem in dcfifo.
-- 08/14/00: Removed DEVICE_FAMILY and OPTIMIZE_FOR_SPEED parameters from
--           scfifo.
-- 08/11/00: dcfifo and scfifo are fully implemented.
-- 08/02/00: Added altdpram.  scfifo and dcfifo are copied from lpm_fifo
--           and lpm_fifo_dc respectively; however, some Altera-specific
--           parameters are not implemented.
----------------------------------------------------------------------------


----------------------------------------------------------------------------
-- altdpram megafunction
----------------------------------------------------------------------------
    
library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_arith.all;
use std.textio.all;

entity altdpram is
    generic
      ( width                  : natural;
        widthad                : natural;
        numwords               : natural := 0;
        lpm_file               : string := "UNUSED";
        lpm_hint               : string := "USE_EAB=ON";
        indata_reg             : string := "UNREGISTERED";
        indata_aclr            : string := "OFF";
        wraddress_reg          : string := "UNREGISTERED";
        wraddress_aclr         : string := "OFF";
        wrcontrol_reg          : string := "UNREGISTERED";
        wrcontrol_aclr         : string := "OFF";
        rdaddress_reg          : string := "UNREGISTERED";
        rdaddress_aclr         : string := "OFF";
        rdcontrol_reg          : string := "UNREGISTERED";
        rdcontrol_aclr         : string := "OFF";
        outdata_reg            : string := "UNREGISTERED";
        outdata_aclr           : string := "OFF";
        intended_device_family : string := "APEX20KE");
    port
      ( wren       : in std_logic := '0';
        data       : in std_logic_vector(width-1 downto 0);
        wraddress  : in std_logic_vector(widthad-1 downto 0);
        inclock    : in std_logic := '0';
        inclocken  : in std_logic := '1';
        rden       : in std_logic := '1';
        rdaddress  : in std_logic_vector(widthad-1 downto 0);
        outclock   : in std_logic := '0';
        outclocken : in std_logic := '1';
        aclr       : in std_logic := '0';
        q          : out std_logic_vector(width-1 downto 0) );

	function int_to_str( value : integer ) return string is
	variable ivalue,index : integer;
	variable digit : integer;
    variable line_no: string(8 downto 1) := "        ";  
	begin
		ivalue := value;
		index := 1;
		while (ivalue > 0) loop
			digit := ivalue MOD 10;
			ivalue := ivalue/10;
			case digit is
				when 0 =>
					line_no(index) := '0';
				when 1 =>
					line_no(index) := '1';
				when 2 =>
					line_no(index) := '2';
				when 3 =>
					line_no(index) := '3';
				when 4 =>
					line_no(index) := '4';
				when 5 =>
					line_no(index) := '5';
				when 6 =>
					line_no(index) := '6';
				when 7 =>
					line_no(index) := '7';
				when 8 =>
					line_no(index) := '8';
				when 9 =>
					line_no(index) := '9';
				when others =>
					ASSERT FALSE
					REPORT "Illegal number!"
					SEVERITY ERROR;
			end case;
			index := index + 1;
		end loop;
		return line_no;
	end;

	function hex_str_to_int( str : string ) return integer is
	variable len : integer := str'length;
	variable ivalue : integer := 0;
	variable digit : integer;
	begin
		for i in len downto 1 loop
			case str(i) is
				when '0' =>
					digit := 0;
				when '1' =>
					digit := 1;
				when '2' =>
					digit := 2;
				when '3' =>
					digit := 3;
				when '4' =>
					digit := 4;
				when '5' =>
					digit := 5;
				when '6' =>
					digit := 6;
				when '7' =>
					digit := 7;
				when '8' =>
					digit := 8;
				when '9' =>
					digit := 9;
				when 'A' =>
					digit := 10;
				when 'a' =>
					digit := 10;
				when 'B' =>
					digit := 11;
				when 'b' =>
					digit := 11;
				when 'C' =>
					digit := 12;
				when 'c' =>
					digit := 12;
				when 'D' =>
					digit := 13;
				when 'd' =>
					digit := 13;
				when 'E' =>
					digit := 14;
				when 'e' =>
					digit := 14;
				when 'F' =>
					digit := 15;
				when 'f' =>
					digit := 15;
				when others =>
					ASSERT FALSE
					REPORT "Illegal character "&  str(i) & "in Intel Hex File! "
					SEVERITY ERROR;
			end case;
			ivalue := ivalue * 16 + digit;
		end loop;
		return ivalue;
	end;

	procedure Shrink_line(L : inout LINE; pos : in integer) is
	subtype nstring is string(1 to pos);
	variable stmp : nstring;
	begin
		if pos >= 1 then
			read(l, stmp);
		end if;
	end;

end altdpram;

architecture behavior of altdpram is

type alt_memory is array((2**WIDTHAD)-1 downto 0) of std_logic_vector(WIDTH-1 downto 0);

signal idata_tmp, idata_reg : std_logic_vector(WIDTH-1 downto 0) := (OTHERS => '0');
signal iq_tmp, iq_reg : std_logic_vector(WIDTH-1 downto 0) := (OTHERS => '0');
signal irdaddress_tmp, irdaddress_reg : std_logic_vector(WIDTHAD-1 downto 0) := (OTHERS => '0');
signal iwraddress_tmp, iwraddress_reg : std_logic_vector(WIDTHAD-1 downto 0) := (OTHERS => '0');
signal iwren_tmp, iwren_reg : std_logic := '0';
signal irden_tmp, irden_reg : std_logic := '0';

begin
    sync: process(data, idata_reg, rden, irden_reg, rdaddress, irdaddress_reg,
                  wren, iwren_reg, wraddress, iwraddress_reg, iq_tmp, iq_reg,
                  aclr)
	begin
        if (rdaddress_reg = "INCLOCK" or rdaddress_reg = "OUTCLOCK") then
            irdaddress_tmp <= irdaddress_reg;
        elsif (rdaddress_reg = "UNREGISTERED") then
            irdaddress_tmp <= rdaddress;
        else
            ASSERT FALSE
            REPORT "Illegal RDADDRESS_REG property value for ALTDPRAM!"
            SEVERITY ERROR;
		end if;
        if (rdcontrol_reg = "INCLOCK" or rdcontrol_reg = "OUTCLOCK") then
            irden_tmp <= irden_reg;
        elsif (rdcontrol_reg = "UNREGISTERED") then
            irden_tmp <= rden;
        else
            ASSERT FALSE
            REPORT "Illegal RDCONTROL_REG property value for ALTDPRAM!"
            SEVERITY ERROR;
		end if;

        if (wraddress_reg = "INCLOCK") then
            iwraddress_tmp <= iwraddress_reg;
        elsif (wraddress_reg = "UNREGISTERED") then
            iwraddress_tmp <= wraddress;
        else
            ASSERT FALSE
            REPORT "Illegal WRADDRESS_REG property value for ALTDPRAM!"
            SEVERITY ERROR;
		end if;
        if (wrcontrol_reg = "INCLOCK") then
            iwren_tmp <= iwren_reg; 
        elsif (wrcontrol_reg = "UNREGISTERED") then
            iwren_tmp <= wren;
        else
            ASSERT FALSE
            REPORT "Illegal WRCONTROL_REG property value for ALTDPRAM!"
            SEVERITY ERROR;
		end if;

        if (indata_reg = "INCLOCK") then
            idata_tmp <= idata_reg;
        elsif (indata_reg = "UNREGISTERED") then
            idata_tmp <= data;
        else
            ASSERT FALSE
            REPORT "Illegal INDATA_REG property value for ALTDPRAM!"
            SEVERITY ERROR;
		end if;
        if (outdata_reg = "OUTCLOCK") then
            q <= iq_reg;
        elsif (outdata_reg = "UNREGISTERED") then
            q <= iq_tmp;
        else
            ASSERT FALSE
            REPORT "Illegal OUTDATA_REG property value for ALTDPRAM!"
            SEVERITY ERROR;
		end if;

        if (aclr = '1') then
            if(indata_aclr = "ON") then
                idata_tmp <= (OTHERS => '0');
            end if;
            if(wraddress_aclr = "ON") then
                iwraddress_tmp <= (OTHERS => '0');
            end if;
            if(wrcontrol_aclr = "ON") then
                iwren_tmp <= '0';
            end if;
            if(rdaddress_aclr = "ON") then
                irdaddress_tmp <= (OTHERS => '0');
            end if;
            if(rdcontrol_aclr = "ON") then
                irden_tmp <= '0';
            end if;
            if(outdata_aclr = "ON") then
                q <= (OTHERS => '0');
            end if;
        end if;
	end process;

    input_reg: process (inclock)
	begin
        if (aclr = '1' and indata_aclr = "ON") then
            idata_reg <= (OTHERS => '0');
        elsif inclock'event and inclock = '1' and inclocken = '1' then
            idata_reg <= data;
        end if;
        if (aclr = '1' and wraddress_aclr = "ON") then
            iwraddress_reg <= (OTHERS => '0');
        elsif inclock'event and inclock = '1' and inclocken = '1' then
            iwraddress_reg <= wraddress;
        end if;
        if (aclr = '1' and wrcontrol_aclr = "ON") then
            iwren_reg <= '0';
        elsif inclock'event and inclock = '1' and inclocken = '1' then
            iwren_reg <= wren;
        end if;
        if (rdaddress_reg = "INCLOCK") then
            if (aclr = '1' and rdaddress_aclr = "ON") then
                irdaddress_reg <= (OTHERS => '0');
            elsif outclock'event and outclock = '1' and outclocken = '1' then
                irdaddress_reg <= rdaddress;
            end if;
        end if;
        if (rdcontrol_reg = "INCLOCK") then
            if (aclr = '1' and rdcontrol_aclr = "ON") then
                irden_reg <= '0';
            elsif outclock'event and outclock = '1' and outclocken = '1' then
                irden_reg <= rden; 
            end if;
        end if;
	end process;

    output_reg: process (outclock)
	begin
        if (aclr = '1' and outdata_aclr = "ON") then
            iq_reg <= (OTHERS => '0');
        elsif outclock'event and outclock = '1' and outclocken = '1' then
            iq_reg <= iq_tmp;
        end if;
        if (rdaddress_reg = "OUTCLOCK") then
            if (aclr = '1' and rdaddress_aclr = "ON") then
                irdaddress_reg <= (OTHERS => '0');
            elsif outclock'event and outclock = '1' and outclocken = '1' then
                irdaddress_reg <= rdaddress;
            end if;
        end if;
        if (rdcontrol_reg = "OUTCLOCK") then
            if (aclr = '1' and rdcontrol_aclr = "ON") then
                irden_reg <= '0';
            elsif outclock'event and outclock = '1' and outclocken = '1' then
                irden_reg <= rden; 
            end if;
        end if;
	end process;

    memory: process(idata_tmp, iwren_tmp, irden_tmp, irdaddress_tmp, iwraddress_tmp)
    variable mem_data : alt_memory;
	variable mem_data_tmp : integer := 0;
    variable mem_init : boolean := false;
    variable i, j, k, lineno : integer := 0;
    variable buf : line;
    variable booval : boolean;
    FILE unused_file : TEXT IS OUT "UNUSED";
    FILE mem_data_file : TEXT IS IN lpm_file;
	variable base, byte, rec_type, datain, addr, checksum: string(2 downto 1);
	variable startadd: string(4 downto 1);
    variable ibase : integer := 0;
    variable ibyte : integer := 0;
    variable istartadd : integer := 0;
    variable check_sum_vec, check_sum_vec_tmp : std_logic_vector(7 downto 0);
	begin
		-- INITIALIZE --
		if NOT(mem_init) then
			-- INITIALIZE TO 0 --
			for i in mem_data'LOW to mem_data'HIGH loop
				mem_data(i) := (OTHERS => '0');
			end loop;

                if (lpm_file = "UNUSED") then
				ASSERT FALSE
				REPORT "Initialization file not found!"
				SEVERITY WARNING;
			else
				WHILE NOT ENDFILE(mem_data_file) loop
					booval := true;
					READLINE(mem_data_file, buf);
					lineno := lineno + 1;
					check_sum_vec := (OTHERS => '0');
					if (buf(buf'LOW) = ':') then
						i := 1;
						shrink_line(buf, i);
						READ(L=>buf, VALUE=>byte, good=>booval);
						if not (booval) then
							ASSERT FALSE
							REPORT "[Line "& int_to_str(lineno) & "]:Illegal Intel Hex Format!"
							SEVERITY ERROR;
						end if;
						ibyte := hex_str_to_int(byte);
						check_sum_vec := unsigned(check_sum_vec) + unsigned(CONV_STD_LOGIC_VECTOR(ibyte, 8));
						READ(L=>buf, VALUE=>startadd, good=>booval);
						if not (booval) then
							ASSERT FALSE
							REPORT "[Line "& int_to_str(lineno) & "]:Illegal Intel Hex Format! "
							SEVERITY ERROR;
						end if;
						istartadd := hex_str_to_int(startadd);
						addr(2) := startadd(4);
						addr(1) := startadd(3);
						check_sum_vec := unsigned(check_sum_vec) + unsigned(CONV_STD_LOGIC_VECTOR(hex_str_to_int(addr), 8));
						addr(2) := startadd(2);
						addr(1) := startadd(1);
						check_sum_vec := unsigned(check_sum_vec) + unsigned(CONV_STD_LOGIC_VECTOR(hex_str_to_int(addr), 8));
						READ(L=>buf, VALUE=>rec_type, good=>booval);
						if not (booval) then
							ASSERT FALSE
							REPORT "[Line "& int_to_str(lineno) & "]:Illegal Intel Hex Format! "
							SEVERITY ERROR;
						end if;
						check_sum_vec := unsigned(check_sum_vec) + unsigned(CONV_STD_LOGIC_VECTOR(hex_str_to_int(rec_type), 8));
					else
						ASSERT FALSE
						REPORT "[Line "& int_to_str(lineno) & "]:Illegal Intel Hex Format! "
						SEVERITY ERROR;
					end if;
					case rec_type is
						when "00"=>     -- Data record
							i := 0;
                            k := (WIDTH + 7) / 8;  -- # of bytes per entry
							while (i < ibyte) loop
								mem_data_tmp := 0;
								for j in 1 to k loop
									READ(L=>buf, VALUE=>datain,good=>booval); -- read in data a byte (2 hex chars) at a time.
									if not (booval) then
										ASSERT FALSE
										REPORT "[Line "& int_to_str(lineno) & "]:Illegal Intel Hex Format! "
										SEVERITY ERROR;
									end if;
									check_sum_vec := unsigned(check_sum_vec) + unsigned(CONV_STD_LOGIC_VECTOR(hex_str_to_int(datain), 8));
									mem_data_tmp := mem_data_tmp * 256 + hex_str_to_int(datain);
								end loop;
								i := i + k;
                                mem_data(ibase + istartadd) := CONV_STD_LOGIC_VECTOR(mem_data_tmp, WIDTH);
								istartadd := istartadd + 1;
							end loop;
						when "01"=>
							exit;
						when "02"=>
							ibase := 0;
							if (ibyte /= 2) then
								ASSERT FALSE
								REPORT "[Line "& int_to_str(lineno) & "]:Illegal Intel Hex Format for record type 02! "
								SEVERITY ERROR;
							end if;
							for i in 0 to (ibyte-1) loop
								READ(L=>buf, VALUE=>base,good=>booval);
								ibase := ibase * 256 + hex_str_to_int(base);
								if not (booval) then
									ASSERT FALSE
									REPORT "[Line "& int_to_str(lineno) & "]:Illegal Intel Hex Format! "
									SEVERITY ERROR;
								end if;
								check_sum_vec := unsigned(check_sum_vec) + unsigned(CONV_STD_LOGIC_VECTOR(hex_str_to_int(base), 8));
							end loop;
							ibase := ibase * 16;
						when OTHERS =>
							ASSERT FALSE
							REPORT "[Line "& int_to_str(lineno) & "]:Illegal record type in Intel Hex File! "
							SEVERITY ERROR;
					end case;
					READ(L=>buf, VALUE=>checksum,good=>booval);
					if not (booval) then
						ASSERT FALSE
						REPORT "[Line "& int_to_str(lineno) & "]:Checksum is missing! "
						SEVERITY ERROR;
					end if;

					check_sum_vec := unsigned(not (check_sum_vec)) + 1 ;
					check_sum_vec_tmp := CONV_STD_LOGIC_VECTOR(hex_str_to_int(checksum),8);

					if (unsigned(check_sum_vec) /= unsigned(check_sum_vec_tmp)) then
						ASSERT FALSE
						REPORT "[Line "& int_to_str(lineno) & "]:Incorrect checksum!"
						SEVERITY ERROR;
					end if;
				end loop;
			end if;
			mem_init := TRUE;

        else -- already initialized

            -- MEMORY FUNCTION --
            if iwren_tmp = '1' then
                mem_data (ieee.std_logic_unsigned.conv_integer(iwraddress_tmp)) := idata_tmp;
            end if;
            if irden_tmp = '1' then
                iq_tmp <= mem_data(ieee.std_logic_unsigned.conv_integer(irdaddress_tmp));
            end if;
        end if; -- if NOT(mem_init)

	end process;

end behavior;


----------------------------------------------------------------------------
-- scfifo megafunction
----------------------------------------------------------------------------

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;
use IEEE.std_logic_unsigned.all;

entity SCFIFO is
    generic
      ( lpm_width               : natural;
        lpm_widthu              : natural;
        lpm_numwords            : natural;
        lpm_showahead           : string := "OFF";
        lpm_hint                : string := "USE_EAB=ON";
        almost_full_value       : natural := 0;
        almost_empty_value      : natural := 0;
        overflow_checking       : string := "ON";
        underflow_checking      : string := "ON";
        allow_rwcycle_when_full : string := "OFF";
        -- parameters that should be stored in lpm_hint
        use_eab                 : string := "ON");
    port
      ( data         : in std_logic_vector(lpm_width-1 downto 0);
        clock        : in std_logic;
        wrreq        : in std_logic;
        rdreq        : in std_logic;
        aclr         : in std_logic := '0';
        sclr         : in std_logic := '0';
        full         : out std_logic;
        almost_full  : out std_logic;
        empty        : out std_logic;
        almost_empty : out std_logic;
        q            : out std_logic_vector(lpm_width-1 downto 0);
        usedw        : out std_logic_vector(lpm_widthu-1 downto 0)
     );
end SCFIFO; 

architecture behavior of SCFIFO is

type lpm_memory is array (lpm_numwords-1 downto 0) of std_logic_vector(lpm_width-1 downto 0);

signal tmp_q : std_logic_vector(lpm_width-1 downto 0) := (OTHERS => '0');
signal read_id, write_id, count_id : integer := 0;
signal empty_flag : std_logic := '1';
signal full_flag : std_logic := '0';
constant ZEROS : std_logic_vector(lpm_width-1 downto 0) := (OTHERS => '0');

begin

	process (clock, aclr)
	variable mem_data : lpm_memory := (OTHERS => ZEROS);
    variable valid_rreq, valid_wreq : boolean;
    variable init : boolean := false;
	begin
        if (not init) then
            if (LPM_SHOWAHEAD /= "ON" and LPM_SHOWAHEAD /= "OFF") then
                ASSERT FALSE
                REPORT "Illegal LPM_SHOWAHEAD property value for SCFIFO!"
                SEVERITY ERROR;
            end if;
            if (UNDERFLOW_CHECKING /= "ON" and UNDERFLOW_CHECKING /= "OFF") then
                ASSERT FALSE
                REPORT "Illegal UNDERFLOW_CHECKING property value for SCFIFO!"
                SEVERITY ERROR;
            end if;
            if (OVERFLOW_CHECKING /= "ON" and OVERFLOW_CHECKING /= "OFF") then
                ASSERT FALSE
                REPORT "Illegal OVERFLOW_CHECKING property value for SCFIFO!"
                SEVERITY ERROR;
            end if;
            if (ALLOW_RWCYCLE_WHEN_FULL /= "ON" and ALLOW_RWCYCLE_WHEN_FULL /= "OFF") then
                ASSERT FALSE
                REPORT "Illegal ALLOW_RWCYCLE_WHEN_FULL property value for SCFIFO!"
                SEVERITY ERROR;
            end if;
            init := true;
        end if; -- not init

		if (aclr = '1') then
			tmp_q <= ZEROS;
			full_flag <= '0';
			empty_flag <= '1';
			read_id <= 0;
			write_id <= 0;
			count_id <= 0;
			if (lpm_showahead = "ON") then
				tmp_q <= mem_data(0);
			end if;
		elsif (clock'event and clock = '1') then
			if (sclr = '1') then
				tmp_q <= mem_data(read_id);
				full_flag <= '0';
				empty_flag <= '1';
				read_id <= 0;
				write_id <= 0;
				count_id <= 0;
				if (lpm_showahead = "ON") then
					tmp_q <= mem_data(0);
				end if;
			else
                valid_rreq := rdreq = '1' and (empty_flag = '0' or underflow_checking = "OFF");
                valid_wreq := wrreq = '1' and (full_flag = '0' or overflow_checking = "OFF" or
                                               (rdreq = '1' and allow_rwcycle_when_full = "ON"));

				----- IF BOTH READ AND WRITE -----
                if (valid_wreq and valid_rreq) then

					mem_data(write_id) := data;
					if (write_id >= lpm_numwords-1) then
						write_id <= 0;
					else
						write_id <= write_id + 1;
					end if;

					tmp_q <= mem_data(read_id);
					if (read_id >= lpm_numwords-1) then
						read_id <= 0;
						if (lpm_showahead = "ON") then
							tmp_q <= mem_data(0);
						end if;
					else
						read_id <= read_id + 1;
						if (lpm_showahead = "ON") then
							tmp_q <= mem_data(read_id+1);
						end if;
					end if;

				----- IF WRITE (ONLY) -----
                elsif (valid_wreq) then

					mem_data(write_id) := data;
					if (lpm_showahead = "ON") then
						tmp_q <= mem_data(read_id);
					end if;
					count_id <= count_id + 1;
					empty_flag <= '0';
					if (count_id >= lpm_numwords-1) then
						full_flag <= '1';
						count_id <= lpm_numwords;
					end if;
					if (write_id >= lpm_numwords-1) then
						write_id <= 0;
					else
						write_id <= write_id + 1;
					end if;

				----- IF READ (ONLY) -----
                elsif (valid_rreq) then

					tmp_q <= mem_data(read_id);
					count_id <= count_id - 1;
					full_flag <= '0';
					if (count_id <= 1) then
						empty_flag <= '1';
						count_id <= 0;
					end if;
					if (read_id >= lpm_numwords-1) then
						read_id <= 0;
						if (lpm_showahead = "ON") then
							tmp_q <= mem_data(0);
						end if;
					else
						read_id <= read_id + 1;
						if (lpm_showahead = "ON") then
							tmp_q <= mem_data(read_id+1);
						end if;
					end if;
				end if;  -- if WRITE and/or READ
			end if;  -- if sclr = '1'
		end if;  -- if aclr = '1'
	end process;

    process (count_id)
    begin
        if (count_id >= almost_full_value) then
            almost_full <= '1';
        else
            almost_full <= '0';
        end if;
        if (count_id < almost_empty_value) then
            almost_empty <= '1';
        else
            almost_empty <= '0';
        end if;
    end process;

	q <= tmp_q;
	full <= full_flag;
	empty <= empty_flag;
	usedw <= conv_std_logic_vector(count_id, lpm_widthu);

end behavior;


----------------------------------------------------------------------------
-- dcfifo_dffpipe (used by dcfifo)
----------------------------------------------------------------------------

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;
use IEEE.std_logic_unsigned.all;

entity DCFIFO_DFFPIPE is
    generic (LPM_DELAY : natural);
    port (D     : in integer;
          Q     : out integer;
          CLOCK : in std_logic;
          ACLR  : in std_logic := '0');
end DCFIFO_DFFPIPE;

architecture behavior of DCFIFO_DFFPIPE is
type delaypipe is array (LPM_DELAY downto 0) of integer;

begin

    process (clock, aclr, d)
    variable intpipe : delaypipe := (OTHERS => 0);
    variable delay : integer := LPM_DELAY-1;
    variable init : boolean := false;
    begin
        if (LPM_DELAY = 0) then
            if (aclr = '1' or not init) then
                q <= 0;
                init := true;
            else
                q <= d;
            end if;
        else
            if (aclr = '1' or not init) then
                for i in LPM_DELAY downto 0 loop
                    intpipe(i) := 0;
                end loop;
                init := true;
                q <= 0;
            end if;
            if (clock'event and clock = '1' and NOW > 0 ns) then
                if (delay > 0) then
                    for i in delay downto 1 loop
                        intpipe(i) := intpipe(i-1);
                    end loop;
                end if;
                intpipe(0) := d;
                q <= intpipe(delay);
            end if;
        end if;
    end process;

end behavior;

----------------------------------------------------------------------------
-- dcfifo_fefifo (used by dcfifo_sync and dcfifo_async)
----------------------------------------------------------------------------

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;
use IEEE.std_logic_unsigned.all;

entity DCFIFO_FEFIFO is
    generic (LPM_WIDTHAD  : natural;
             LPM_NUMWORDS : natural;
             UNDERFLOW_CHECKING : string := "ON";
             OVERFLOW_CHECKING : string := "ON";
             LPM_MODE     : string );
    port (USEDW_IN : in std_logic_vector(LPM_WIDTHAD-1 downto 0);
          WREQ     : in std_logic := 'Z';
          RREQ     : in std_logic := 'Z';
          EMPTY    : out std_logic;
          FULL     : out std_logic;
          CLOCK    : in std_logic;
          ACLR     : in std_logic := '0');
end DCFIFO_FEFIFO;

architecture behavior of DCFIFO_FEFIFO is
signal usedw : std_logic_vector(LPM_WIDTHAD-1 downto 0);
signal sm_empty : std_logic_vector(1 downto 0) := "00";
signal lrreq : std_logic := '0'; -- DFF;
signal i_empty : std_logic := '1';
signal i_full : std_logic := '0';
signal valid_rreq : std_logic;

begin

    -- Initialization
    process (clock, aclr)
    variable init : boolean := false;
    begin
        if (not init) then
            if (LPM_MODE /= "READ" and LPM_MODE /= "WRITE") then
                ASSERT FALSE
                REPORT "Illegal LPM_MODE property value for DCFIFO_FEFIFO!"
                SEVERITY ERROR;
            end if;
            if (UNDERFLOW_CHECKING /= "ON" and UNDERFLOW_CHECKING /= "OFF") then
                ASSERT FALSE
                REPORT "Illegal UNDERFLOW_CHECKING property value for DCFIFO_FEFIFO!"
                SEVERITY ERROR;
            end if;
            if (OVERFLOW_CHECKING /= "ON" and OVERFLOW_CHECKING /= "OFF") then
                ASSERT FALSE
                REPORT "Illegal OVERFLOW_CHECKING property value for DCFIFO_FEFIFO!"
                SEVERITY ERROR;
            end if;
            init := true;
        end if; -- not init
    end process;

    valid_rreq <= rreq when underflow_checking = "OFF" else
                  rreq and not i_empty;

    process (clock, aclr)
    begin
        if (aclr = '1') then
            lrreq <= '0';
        elsif (clock'event and clock = '1' and NOW > 0 ns) then
            lrreq <= valid_rreq;
        end if;
    end process;

    process (clock, aclr)
    variable almost_full : integer := 0;
    begin
        if (aclr = '1') then
            i_full <= '0';
        elsif (clock'event and clock = '1' and NOW > 0 ns) then
            if (lpm_numwords >= 3) then
                almost_full := lpm_numwords-3;
            end if;
            if (unsigned(usedw) >= almost_full) then
                i_full <= '1';
            else
                i_full <= '0';
            end if;
        end if;
    end process;

    process (clock, aclr)
    variable local_sm_empty : std_logic_vector(1 downto 0) := sm_empty;
    variable usedw_is_1 : boolean;
    begin
        local_sm_empty := sm_empty;
        if (aclr = '1') then
            local_sm_empty := "00";
        elsif (clock'event and clock = '1' and NOW > 0 ns) then
            if (lpm_mode = "READ") then
                case sm_empty is
                    when "00" =>                    -- state_empty
                        if (usedw /= 0) then
                            local_sm_empty := "01";
                        end if;
                    when "01" =>                    -- state_non_empty
                        usedw_is_1 := (usedw = 1 and lrreq = '0') or (usedw = 2 and lrreq = '1');
                        if (rreq = '1' and usedw_is_1) then
                            local_sm_empty := "10";
                        end if;
                    when "10" =>                    -- state_emptywait
                        if (usedw > 1) then
                            local_sm_empty := "01";
                        else
                            local_sm_empty := "00";
                        end if;
                    when others =>
                        -- INTERNAL ERROR
                end case;
            elsif (lpm_mode = "WRITE") then
                case sm_empty is
                    when "00" =>                    -- state_empty
                        if (wreq = '1') then
                            local_sm_empty := "01";
                        end if;
                    when "01" =>                    -- state_one
                        if (wreq = '0') then
                            local_sm_empty := "11";
                        end if;
                    when "11" =>                    -- state_non_empty
                        if (wreq = '1') then
                            local_sm_empty := "01";
                        elsif (usedw = 0) then
                            local_sm_empty := "00";
                        end if;
                    when others =>
                        -- INTERNAL ERROR
                end case;
            end if;
        end if;
        sm_empty <= local_sm_empty;
        i_empty <= not local_sm_empty(0);
    end process;

    usedw <= usedw_in;
    empty <= i_empty;
    full <= i_full;

end behavior;


----------------------------------------------------------------------------
-- dcfifo_async (used by dcfifo)
----------------------------------------------------------------------------

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;
use IEEE.std_logic_unsigned.all;

entity DCFIFO_ASYNC is
    generic
      ( lpm_width               : natural;
        lpm_widthu              : natural;
        lpm_numwords            : natural;
        lpm_showahead           : string := "OFF";
        lpm_hint                : string := "USE_EAB=ON";
        overflow_checking       : string := "ON";
        underflow_checking      : string := "ON";
        delay_rdusedw           : natural := 1;
        delay_wrusedw           : natural := 1;
        rdsync_delaypipe        : natural := 3;
        wrsync_delaypipe        : natural := 3;
        -- parameters that should be stored in lpm_hint
        use_eab                 : string := "ON";
        clocks_are_synchronized : string := "FALSE");
    port
      ( data    : in std_logic_vector(lpm_width-1 downto 0);
        rdclk   : in std_logic;
        wrclk   : in std_logic;
        wrreq   : in std_logic;
        rdreq   : in std_logic;
        aclr    : in std_logic := '0';
        rdfull  : out std_logic;
        wrfull  : out std_logic;
        wrempty : out std_logic;
        rdempty : out std_logic;
        q       : out std_logic_vector(lpm_width-1 downto 0);
        rdusedw : out std_logic_vector(lpm_widthu-1 downto 0);
        wrusedw : out std_logic_vector(lpm_widthu-1 downto 0));
end DCFIFO_ASYNC; 

architecture behavior of DCFIFO_ASYNC is

type lpm_memory is array (2**lpm_widthu-1 downto 0) of std_logic_vector(lpm_width-1 downto 0);

signal i_q : std_logic_vector(lpm_width-1 downto 0) := (OTHERS => '0');
signal i_rdptr, i_wrptr, i_rdptrrg, i_wrdelaycycle : integer := 0;
signal i_ws_nbrp, i_rs_nbwp, i_ws_dbrp, i_rs_dbwp : integer := 0;
signal i_wr_udwn, i_rd_udwn, i_wr_dbuw, i_rd_dbuw : integer := 0;
signal i_rdempty, i_wrempty : std_logic := '1';
signal i_rdfull, i_wrfull : std_logic := '0';
signal i_rdusedw, i_wrusedw : integer := 0;
signal i_rden, i_wren, i_rdenclock : std_logic := '0';
signal slv_wr_dbuw, slv_rd_dbuw : std_logic_vector(lpm_widthu-1 downto 0);

signal i_data_tmp, i_data_reg : std_logic_vector(lpm_width-1 downto 0);
signal i_q_tmp, i_q_reg : std_logic_vector(lpm_width-1 downto 0) := (OTHERS => '0');
signal i_rdptr_tmp, i_wrptr_tmp, i_wrptr_reg : integer := 0;
signal i_wren_tmp, i_wren_reg : std_logic := '0';

constant ZEROS : std_logic_vector(lpm_width-1 downto 0) := (OTHERS => '0');
constant GRAY_DELAYPIPE : integer := 1;
constant WRUSEDW_DELAYPIPE : integer := 1;  -- delayed usedw to compute empty/full
constant RDUSEDW_DELAYPIPE : integer := 1;  -- delayed usedw to compute empty/full

component DCFIFO_FEFIFO
    generic (LPM_WIDTHAD : natural;
             LPM_NUMWORDS : natural;
             UNDERFLOW_CHECKING : string := "ON";
             OVERFLOW_CHECKING : string := "ON";
             LPM_MODE : string);
    port (USEDW_IN : in std_logic_vector(LPM_WIDTHAD-1 downto 0);
          WREQ : in std_logic := 'Z';
          RREQ : in std_logic := 'Z';
          EMPTY : out std_logic;
          FULL : out std_logic;
          CLOCK : in std_logic;
          ACLR : in std_logic := '0');
end component;

component DCFIFO_DFFPIPE
    generic (LPM_DELAY : natural);
    port (D : in integer;
          Q : out integer;
          CLOCK : in std_logic;
          ACLR : in std_logic := '0');
end component;

begin

    -- Initialization
    process (wrclk, rdclk, aclr)
    variable init : boolean := false;
    begin
        if (not init) then
            if (LPM_SHOWAHEAD /= "ON" and LPM_SHOWAHEAD /= "OFF") then
                ASSERT FALSE
                REPORT "Illegal LPM_SHOWAHEAD property value for DCFIFO!"
                SEVERITY ERROR;
            end if;
            if (UNDERFLOW_CHECKING /= "ON" and UNDERFLOW_CHECKING /= "OFF") then
                ASSERT FALSE
                REPORT "Illegal UNDERFLOW_CHECKING property value for DCFIFO!"
                SEVERITY ERROR;
            end if;
            if (OVERFLOW_CHECKING /= "ON" and OVERFLOW_CHECKING /= "OFF") then
                ASSERT FALSE
                REPORT "Illegal OVERFLOW_CHECKING property value for DCFIFO!"
                SEVERITY ERROR;
            end if;
            init := true;
        end if; -- not init
    end process;


    ----------
    -- FIFOram
    ----------

    i_rden <= rdreq when underflow_checking = "OFF" else
              rdreq and not i_rdempty;
    i_wren <= wrreq when overflow_checking = "OFF" else
              wrreq and not i_wrfull;

    FIFOram_sync: process (i_data_reg, i_q_tmp, i_q_reg, aclr,
                           i_rdptr, i_wren_reg, i_wrptr_reg)
    begin
        if (aclr = '1') then
            i_wrptr_tmp <= 0;
            i_rdptr_tmp <= 0;
            i_wren_tmp <= '0';
            i_data_tmp <= (OTHERS => '0');
            if (LPM_SHOWAHEAD = "ON") then
                i_q <= i_q_tmp;
            else
                i_q <= (OTHERS => '0');
            end if;
        else
            i_wrptr_tmp <= i_wrptr_reg;
            i_rdptr_tmp <= i_rdptr;
            i_wren_tmp <= i_wren_reg;
            i_data_tmp <= i_data_reg;
            if (LPM_SHOWAHEAD = "ON") then
                i_q <= i_q_tmp;
            else
                i_q <= i_q_reg;
            end if;
        end if;
    end process;

    FIFOram_wrclk: process (wrclk, aclr)
    begin
        if (aclr = '1') then
            i_data_reg <= (OTHERS => '0');
            i_wrptr_reg <= 0;
            i_wren_reg <= '0';
        elsif (wrclk'event and wrclk = '1' and NOW > 0 ns) then
            i_data_reg <= data;
            i_wrptr_reg <= i_wrptr;
            i_wren_reg <= i_wren;
        end if;
    end process;

    FIFOram_rdclk: process (rdclk, aclr)
    begin
        if (aclr = '1') then
            i_q_reg <= (OTHERS => '0');
        elsif (rdclk'event and rdclk = '1' and i_rden = '1' and NOW > 0 ns) then
            i_q_reg <= i_q_tmp;
        end if;
    end process;

    FIFOram_memory: process (i_data_tmp, i_wren_tmp, i_wrptr_tmp, i_rdptr_tmp, wrclk)
    variable mem_data : lpm_memory := (OTHERS => ZEROS);
    variable init : boolean := false;
    begin
        if (not init) then
            for i in lpm_numwords-1 downto 0 loop
                mem_data(i) := ZEROS;
            end loop;
            init := true;
        end if;
        if wrclk'event and i_wren_tmp='1' and
            ((wrclk='1' and USE_EAB="OFF") or
             (wrclk='0' and USE_EAB="ON")) then
            mem_data(i_wrptr_tmp) := i_data_tmp;
        end if;
        i_q_tmp <= mem_data(i_rdptr_tmp);
    end process;


    -----------
    -- Counters
    -----------

    rdptr: process (rdclk, aclr)
    begin
        if (aclr = '1') then
            i_rdptr <= 0;
        elsif (rdclk'event and rdclk = '1' and i_rden = '1' and NOW > 0 ns) then
            if (i_rdptr < 2**lpm_widthu-1) then
                i_rdptr <= i_rdptr + 1;
            else
                i_rdptr <= 0;
            end if;
        end if;
    end process;

    wrptr: process (wrclk, aclr)
    begin
        if (aclr = '1') then
            i_wrptr <= 0;
        elsif (wrclk'event and wrclk = '1' and i_wren = '1' and NOW > 0 ns) then
            if (i_wrptr < 2**lpm_widthu-1) then
                i_wrptr <= i_wrptr + 1;
            else
                i_wrptr <= 0;
            end if;
        end if;
    end process;


    ---------------------
    -- Delays & DFF Pipes
    ---------------------

    process (rdclk)
    begin
        if (rdclk = '0') then
            i_rdenclock <= '0';
        elsif (rdclk = '1' and i_rden = '1') then
            i_rdenclock <= '1';
        end if;
    end process;

    RDPTR_D:    DCFIFO_DFFPIPE
                    generic map (LPM_DELAY => 0)
                    port map (D => i_rdptr, Q => i_rdptrrg,
                              CLOCK => i_rdenclock, ACLR => aclr);

    WRPTR_D:    DCFIFO_DFFPIPE
                    generic map (LPM_DELAY => 1)
                    port map (D => i_wrptr, Q => i_wrdelaycycle,
                              CLOCK => wrclk, ACLR => aclr);

    WS_NBRP:    DCFIFO_DFFPIPE
                    generic map (LPM_DELAY => WRSYNC_DELAYPIPE)
                    port map (D => i_rdptrrg, Q => i_ws_nbrp,
                              CLOCK => wrclk, ACLR => aclr);

    RS_NBWP:    DCFIFO_DFFPIPE
                    generic map (LPM_DELAY => RDSYNC_DELAYPIPE)
                    port map (D => i_wrdelaycycle, Q => i_rs_nbwp,
                              CLOCK => rdclk, ACLR => aclr);

    WS_DBRP:    DCFIFO_DFFPIPE
                    generic map (LPM_DELAY => GRAY_DELAYPIPE)
                    port map (D => i_ws_nbrp, Q => i_ws_dbrp,
                              CLOCK => wrclk, ACLR => aclr);

    RS_DBWP:    DCFIFO_DFFPIPE
                    generic map (LPM_DELAY => GRAY_DELAYPIPE)
                    port map (D => i_rs_nbwp, Q => i_rs_dbwp,
                              CLOCK => rdclk, ACLR => aclr);

    process (i_wrptr, i_ws_dbrp)
    begin
        i_wr_udwn <= i_wrptr - i_ws_dbrp;
    end process;
            
    process (i_rdptr, i_rs_dbwp)
    begin
        i_rd_udwn <= i_rs_dbwp - i_rdptr;
    end process;
            
    WR_USEDW:   DCFIFO_DFFPIPE
                    generic map (LPM_DELAY => DELAY_WRUSEDW)
                    port map (D => i_wr_udwn, Q => i_wrusedw,
                              CLOCK => wrclk, ACLR => aclr);

    RD_USEDW:   DCFIFO_DFFPIPE
                    generic map (LPM_DELAY => DELAY_RDUSEDW)
                    port map (D => i_rd_udwn, Q => i_rdusedw,
                              CLOCK => rdclk, ACLR => aclr);

    WR_DBUW:    DCFIFO_DFFPIPE
                    generic map (LPM_DELAY => WRUSEDW_DELAYPIPE)
                    port map (D => i_wr_udwn, Q => i_wr_dbuw,
                              CLOCK => wrclk, ACLR => aclr);

    RD_DBUW:    DCFIFO_DFFPIPE
                    generic map (LPM_DELAY => RDUSEDW_DELAYPIPE)
                    port map (D => i_rd_udwn, Q => i_rd_dbuw,
                              CLOCK => rdclk, ACLR => aclr);


    -------------
    -- Full/Empty
    -------------

    slv_wr_dbuw <= conv_std_logic_vector(i_wr_dbuw, LPM_WIDTHU);
    slv_rd_dbuw <= conv_std_logic_vector(i_rd_dbuw, LPM_WIDTHU);

    WR_FE:  DCFIFO_FEFIFO
                generic map (LPM_WIDTHAD => LPM_WIDTHU,
                             LPM_NUMWORDS => LPM_NUMWORDS,
                             UNDERFLOW_CHECKING => UNDERFLOW_CHECKING,
                             OVERFLOW_CHECKING => OVERFLOW_CHECKING,
                             LPM_MODE => "WRITE")
                port map (USEDW_IN => slv_wr_dbuw, WREQ => wrreq,
                          CLOCK => wrclk, ACLR => aclr,
                          EMPTY => i_wrempty, FULL => i_wrfull);

    RD_FE:  DCFIFO_FEFIFO
                generic map (LPM_WIDTHAD => LPM_WIDTHU,
                             LPM_NUMWORDS => LPM_NUMWORDS,
                             UNDERFLOW_CHECKING => UNDERFLOW_CHECKING,
                             OVERFLOW_CHECKING => OVERFLOW_CHECKING,
                             LPM_MODE => "READ")
                port map (USEDW_IN => slv_rd_dbuw, RREQ => rdreq,
                          CLOCK => rdclk, ACLR => aclr,
                          EMPTY => i_rdempty, FULL => i_rdfull);

    ----------
    -- Outputs
    ----------

    q <= i_q;
    wrfull <= i_wrfull;
    wrempty <= i_wrempty;
    rdfull <= i_rdfull;
    rdempty <= i_rdempty;
    wrusedw <= conv_std_logic_vector(i_wrusedw, LPM_WIDTHU);
    rdusedw <= conv_std_logic_vector(i_rdusedw, LPM_WIDTHU);

end behavior;


----------------------------------------------------------------------------
-- dcfifo_sync (used by dcfifo)
----------------------------------------------------------------------------

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;
use IEEE.std_logic_unsigned.all;

entity DCFIFO_SYNC is
    generic
      ( lpm_width               : natural;
        lpm_widthu              : natural;
        lpm_numwords            : natural;
        lpm_showahead           : string := "OFF";
        overflow_checking       : string := "ON";
        underflow_checking      : string := "ON";
        use_eab                 : string := "ON");
    port
      ( data    : in std_logic_vector(lpm_width-1 downto 0);
        rdclk   : in std_logic;
        wrclk   : in std_logic;
        wrreq   : in std_logic;
        rdreq   : in std_logic;
        aclr    : in std_logic := '0';
        rdfull  : out std_logic;
        wrfull  : out std_logic;
        wrempty : out std_logic;
        rdempty : out std_logic;
        q       : out std_logic_vector(lpm_width-1 downto 0);
        rdusedw : out std_logic_vector(lpm_widthu-1 downto 0);
        wrusedw : out std_logic_vector(lpm_widthu-1 downto 0));
end DCFIFO_SYNC;

architecture behavior of DCFIFO_SYNC is

type lpm_memory is array (2**lpm_widthu-1 downto 0) of std_logic_vector(lpm_width-1 downto 0);

signal i_q : std_logic_vector(lpm_width-1 downto 0) := (OTHERS => '0');
signal i_rdptr, i_wrptr : integer := 0;
signal i_rdptr_s, i_wrptr_r, i_wrptr_s : integer := 0;
signal i_rdempty, i_wrempty : std_logic := '1';
signal i_rdfull, i_wrfull : std_logic := '0';
signal i_rdusedw, i_wrusedw : integer := 0;
signal i_rden, i_wren : std_logic := '0';
signal i_cnt_mod : integer;

signal i_data_tmp, i_data_reg : std_logic_vector(lpm_width-1 downto 0);
signal i_q_tmp, i_q_reg : std_logic_vector(lpm_width-1 downto 0) := (OTHERS => '0');
signal i_rdptr_tmp, i_wrptr_tmp, i_wrptr_reg : integer := 0;
signal i_wren_tmp, i_wren_reg : std_logic := '0';

constant ZEROS : std_logic_vector(lpm_width-1 downto 0) := (OTHERS => '0');
constant GRAY_DELAYPIPE : integer := 1;
constant WRUSEDW_DELAYPIPE : integer := 1;  -- delayed usedw to compute empty/full
constant RDUSEDW_DELAYPIPE : integer := 1;  -- delayed usedw to compute empty/full

component DCFIFO_FEFIFO
    generic (LPM_WIDTHAD : natural;
             LPM_NUMWORDS : natural;
             UNDERFLOW_CHECKING : string := "ON";
             OVERFLOW_CHECKING : string := "ON";
             LPM_MODE : string);
    port (USEDW_IN : in std_logic_vector(LPM_WIDTHAD-1 downto 0);
          WREQ : in std_logic := 'Z';
          RREQ : in std_logic := 'Z';
          EMPTY : out std_logic;
          FULL : out std_logic;
          CLOCK : in std_logic;
          ACLR : in std_logic := '0');
end component;

component DCFIFO_DFFPIPE
    generic (LPM_DELAY : natural);
    port (D : in integer;
          Q : out integer;
          CLOCK : in std_logic;
          ACLR : in std_logic := '0');
end component;

begin

    -- Initialization
    process (wrclk, rdclk, aclr)
    variable init : boolean := false;
    begin
        if (not init) then
            if (LPM_SHOWAHEAD /= "ON" and LPM_SHOWAHEAD /= "OFF") then
                ASSERT FALSE
                REPORT "Illegal LPM_SHOWAHEAD property value for DCFIFO!"
                SEVERITY ERROR;
            end if;
            if (UNDERFLOW_CHECKING /= "ON" and UNDERFLOW_CHECKING /= "OFF") then
                ASSERT FALSE
                REPORT "Illegal UNDERFLOW_CHECKING property value for DCFIFO!"
                SEVERITY ERROR;
            end if;
            if (OVERFLOW_CHECKING /= "ON" and OVERFLOW_CHECKING /= "OFF") then
                ASSERT FALSE
                REPORT "Illegal OVERFLOW_CHECKING property value for DCFIFO!"
                SEVERITY ERROR;
            end if;
            if (LPM_NUMWORDS > 2**LPM_WIDTHU) then
                ASSERT FALSE
                REPORT "LPM_NUMWORDS must be less than or equal to 2**LPM_WIDTHU!"
                SEVERITY ERROR;
            end if;

            if lpm_numwords = 2**lpm_widthu then
                i_cnt_mod <= 2**(lpm_widthu+1);
            else
                i_cnt_mod <= 2**lpm_widthu;
            end if;

            init := true;
        end if; -- not init
    end process;


    ----------
    -- FIFOram
    ----------

    i_rden <= rdreq when (underflow_checking = "OFF") else
              rdreq and not i_rdempty;
    i_wren <= wrreq when (overflow_checking = "OFF") else
              wrreq and not i_wrfull;

    FIFOram_sync: process (i_data_reg, i_q_tmp, i_q_reg, aclr,
                           i_rdptr, i_wren_reg, i_wrptr_reg)
    begin
        if (aclr = '1') then
            i_wrptr_tmp <= 0;
            i_rdptr_tmp <= 0;
            i_wren_tmp <= '0';
            i_data_tmp <= (OTHERS => '0');
            if (LPM_SHOWAHEAD = "ON") then
                i_q <= i_q_tmp;
            else
                i_q <= (OTHERS => '0');
            end if;
        else
            i_wrptr_tmp <= i_wrptr_reg;
            i_rdptr_tmp <= i_rdptr;
            i_wren_tmp <= i_wren_reg;
            i_data_tmp <= i_data_reg;
            if (LPM_SHOWAHEAD = "ON") then
                i_q <= i_q_tmp;
            else
                i_q <= i_q_reg;
            end if;
        end if;
    end process;

    FIFOram_wrclk: process (wrclk, aclr)
    begin
        if (aclr = '1') then
            i_data_reg <= (OTHERS => '0');
            i_wrptr_reg <= 0;
            i_wren_reg <= '0';
        elsif (wrclk'event and wrclk = '1' and NOW > 0 ns) then
            i_data_reg <= data;
            i_wrptr_reg <= i_wrptr;
            i_wren_reg <= i_wren;
        end if;
    end process;

    FIFOram_rdclk: process (rdclk, aclr)
    begin
        if (aclr = '1') then
            i_q_reg <= (OTHERS => '0');
        elsif (rdclk'event and rdclk = '1' and i_rden = '1' and NOW > 0 ns) then
            i_q_reg <= i_q_tmp;
        end if;
    end process;

    FIFOram_memory: process (i_data_tmp, i_wren_tmp, i_wrptr_tmp, i_rdptr_tmp, wrclk)
    variable mem_data : lpm_memory := (OTHERS => ZEROS);
    variable init : boolean := false;
    begin
        if (not init) then
            for i in lpm_numwords-1 downto 0 loop
                mem_data(i) := ZEROS;
            end loop;
            init := true;
        end if;
        if wrclk'event and i_wren_tmp='1' and NOW > 0 ns and
            ((wrclk='1' and USE_EAB="OFF") or
             (wrclk='0' and USE_EAB="ON")) then
            mem_data(i_wrptr_tmp mod (2**lpm_widthu)) := i_data_tmp;
        end if;
        i_q_tmp <= mem_data(i_rdptr_tmp mod (2**lpm_widthu));
    end process;


    -----------
    -- Counters
    -----------

    rdptr: process (rdclk, aclr)
    begin
        if (aclr = '1') then
            i_rdptr <= 0;
        elsif (rdclk'event and rdclk = '1' and i_rden = '1' and NOW > 0 ns) then
            if (i_rdptr < i_cnt_mod-1) then
                i_rdptr <= i_rdptr + 1;
            else
                i_rdptr <= 0;
            end if;
        end if;
    end process;

    wrptr: process (wrclk, aclr)
    begin
        if (aclr = '1') then
            i_wrptr <= 0;
        elsif (wrclk'event and wrclk = '1' and i_wren = '1' and NOW > 0 ns) then
            if (i_wrptr < i_cnt_mod-1) then
                i_wrptr <= i_wrptr + 1;
            else
                i_wrptr <= 0;
            end if;
        end if;
    end process;


    ---------
    -- Delays
    ---------

    RDPTR_D:    DCFIFO_DFFPIPE
                    generic map (LPM_DELAY => 1)
                    port map (D => i_rdptr, Q => i_rdptr_s,
                              CLOCK => wrclk, ACLR => aclr);

    WRPTR_D:    DCFIFO_DFFPIPE
                    generic map (LPM_DELAY => 1)
                    port map (D => i_wrptr, Q => i_wrptr_r,
                              CLOCK => wrclk, ACLR => aclr);

    WRPTR_E:    DCFIFO_DFFPIPE
                    generic map (LPM_DELAY => 1)
                    port map (D => i_wrptr_r, Q => i_wrptr_s,
                              CLOCK => rdclk, ACLR => aclr);


    -------------------
    -- Usedw/Full/Empty
    -------------------

    process (i_wrptr, i_rdptr_s)
    begin        
        i_wrusedw <= (i_wrptr - i_rdptr_s) mod i_cnt_mod;
    end process;
            
    process (i_rdptr, i_wrptr_s)
    begin
        i_rdusedw <= (i_wrptr_s - i_rdptr) mod i_cnt_mod;
    end process;

    process (i_wrusedw)
    begin
        i_wrfull <= '0';
        i_wrempty <= '0';

        if ((lpm_numwords = 2**lpm_widthu) and (i_wrusedw >= 2**lpm_widthu)) or
           ((lpm_numwords < 2**lpm_widthu) and (i_wrusedw = lpm_numwords)) then
            i_wrfull <= '1';
        end if;
        if (i_wrusedw = 0) then
            i_wrempty <= '1';
        end if;
    end process;

    process (i_rdusedw)
    begin
        i_rdfull <= '0';
        i_rdempty <= '0';

        if ((lpm_numwords = 2**lpm_widthu) and (i_rdusedw >= 2**lpm_widthu)) or
           ((lpm_numwords < 2**lpm_widthu) and (i_rdusedw = lpm_numwords)) then
            i_rdfull <= '1';
        end if;
        if (i_rdusedw = 0) then
            i_rdempty <= '1';
        end if;
    end process;


    ----------
    -- Outputs
    ----------

    q <= i_q;
    wrfull <= i_wrfull;
    wrempty <= i_wrempty;
    rdfull <= i_rdfull;
    rdempty <= i_rdempty;
    wrusedw <= conv_std_logic_vector(i_wrusedw, LPM_WIDTHU);
    rdusedw <= conv_std_logic_vector(i_rdusedw, LPM_WIDTHU);

end behavior;


----------------------------------------------------------------------------
-- dcfifo megafunction
----------------------------------------------------------------------------

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;
use IEEE.std_logic_unsigned.all;

entity DCFIFO is
    generic
      ( lpm_width               : natural;
        lpm_widthu              : natural;
        lpm_numwords            : natural;
        lpm_showahead           : string := "OFF";
        lpm_hint                : string := "USE_EAB=ON";
        overflow_checking       : string := "ON";
        underflow_checking      : string := "ON";
        delay_rdusedw           : natural := 1;
        delay_wrusedw           : natural := 1;
        rdsync_delaypipe        : natural := 3;
        wrsync_delaypipe        : natural := 3;
        -- parameters that should be stored in lpm_hint
        use_eab                 : string := "ON";
        clocks_are_synchronized : string := "FALSE");
    port
      ( data    : in std_logic_vector(lpm_width-1 downto 0);
        rdclk   : in std_logic;
        wrclk   : in std_logic;
        wrreq   : in std_logic;
        rdreq   : in std_logic;
        aclr    : in std_logic := '0';
        rdfull  : out std_logic;
        wrfull  : out std_logic;
        wrempty : out std_logic;
        rdempty : out std_logic;
        q       : out std_logic_vector(lpm_width-1 downto 0);
        rdusedw : out std_logic_vector(lpm_widthu-1 downto 0);
        wrusedw : out std_logic_vector(lpm_widthu-1 downto 0));
end DCFIFO; 

architecture behavior of DCFIFO is

signal i_q_s, i_q_a : std_logic_vector(lpm_width-1 downto 0);
signal i_wrusedw_s, i_wrusedw_a : std_logic_vector(lpm_widthu-1 downto 0);
signal i_rdusedw_s, i_rdusedw_a : std_logic_vector(lpm_widthu-1 downto 0);
signal i_wrfull_s, i_wrfull_a, i_rdfull_s, i_rdfull_a : std_logic;
signal i_wrempty_s, i_wrempty_a, i_rdempty_s, i_rdempty_a : std_logic;

component DCFIFO_ASYNC
    generic
      ( lpm_width               : natural;
        lpm_widthu              : natural;
        lpm_numwords            : natural;
        lpm_showahead           : string := "OFF";
        overflow_checking       : string := "ON";
        underflow_checking      : string := "ON";
        delay_rdusedw           : natural := 1;
        delay_wrusedw           : natural := 1;
        rdsync_delaypipe        : natural := 3;
        wrsync_delaypipe        : natural := 3;
        use_eab                 : string := "ON");
    port
      ( data    : in std_logic_vector(lpm_width-1 downto 0);
        rdclk   : in std_logic;
        wrclk   : in std_logic;
        wrreq   : in std_logic;
        rdreq   : in std_logic;
        aclr    : in std_logic := '0';
        rdfull  : out std_logic;
        wrfull  : out std_logic;
        wrempty : out std_logic;
        rdempty : out std_logic;
        q       : out std_logic_vector(lpm_width-1 downto 0);
        rdusedw : out std_logic_vector(lpm_widthu-1 downto 0);
        wrusedw : out std_logic_vector(lpm_widthu-1 downto 0));
end component;

component DCFIFO_SYNC
    generic
      ( lpm_width               : natural;
        lpm_widthu              : natural;
        lpm_numwords            : natural;
        lpm_showahead           : string := "OFF";
        overflow_checking       : string := "ON";
        underflow_checking      : string := "ON";
        use_eab                 : string := "ON");
    port
      ( data    : in std_logic_vector(lpm_width-1 downto 0);
        rdclk   : in std_logic;
        wrclk   : in std_logic;
        wrreq   : in std_logic;
        rdreq   : in std_logic;
        aclr    : in std_logic := '0';
        rdfull  : out std_logic;
        wrfull  : out std_logic;
        wrempty : out std_logic;
        rdempty : out std_logic;
        q       : out std_logic_vector(lpm_width-1 downto 0);
        rdusedw : out std_logic_vector(lpm_widthu-1 downto 0);
        wrusedw : out std_logic_vector(lpm_widthu-1 downto 0));
end component;

begin

    ASYNC:  DCFIFO_ASYNC
                generic map (lpm_width => lpm_width,
                             lpm_widthu => lpm_widthu,
                             lpm_numwords => lpm_numwords,
                             lpm_showahead => lpm_showahead,
                             overflow_checking => overflow_checking,
                             underflow_checking => underflow_checking,
                             delay_rdusedw => delay_rdusedw,
                             delay_wrusedw => delay_wrusedw,
                             rdsync_delaypipe => rdsync_delaypipe,
                             wrsync_delaypipe => wrsync_delaypipe,
                             use_eab => use_eab)
                port map (data => data, rdclk => rdclk, wrclk => wrclk,
                          wrreq => wrreq, rdreq => rdreq, aclr => aclr,
                          rdfull => i_rdfull_a, wrfull => i_wrfull_a,
                          rdempty => i_rdempty_a, wrempty => i_wrempty_a,
                          rdusedw => i_rdusedw_a, wrusedw => i_wrusedw_a,
                          q => i_q_a);

    SYNC:   DCFIFO_SYNC
                generic map (lpm_width => lpm_width,
                             lpm_widthu => lpm_widthu,
                             lpm_numwords => lpm_numwords,
                             lpm_showahead => lpm_showahead,
                             overflow_checking => overflow_checking,
                             underflow_checking => underflow_checking,
                             use_eab => use_eab)
                port map (data => data, rdclk => rdclk, wrclk => wrclk,
                          wrreq => wrreq, rdreq => rdreq, aclr => aclr,
                          rdfull => i_rdfull_s, wrfull => i_wrfull_s,
                          rdempty => i_rdempty_s, wrempty => i_wrempty_s,
                          rdusedw => i_rdusedw_s, wrusedw => i_wrusedw_s,
                          q => i_q_s);

    rdfull <= i_rdfull_a when clocks_are_synchronized = "FALSE" else
              i_rdfull_s;
    wrfull <= i_wrfull_a when clocks_are_synchronized = "FALSE" else
              i_wrfull_s;
    rdempty <= i_rdempty_a when clocks_are_synchronized = "FALSE" else
               i_rdempty_s;
    wrempty <= i_wrempty_a when clocks_are_synchronized = "FALSE" else
               i_wrempty_s;
    rdusedw <= i_rdusedw_a when clocks_are_synchronized = "FALSE" else
               i_rdusedw_s;
    wrusedw <= i_wrusedw_a when clocks_are_synchronized = "FALSE" else
               i_wrusedw_s;
    q <= i_q_a when clocks_are_synchronized = "FALSE" else
         i_q_s;

end behavior;
