/*
 * Copyright (c) 2015, 2017, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License, version 2.0, as
 * published by the Free Software Foundation.
 *
 * This program is also distributed with certain software (including
 * but not limited to OpenSSL) that is licensed under separate terms,
 * as designated in a particular file or component or in included license
 * documentation.  The authors of MySQL hereby grant you an
 * additional permission to link the program and your derivative works
 * with the separately licensed software that they have included with
 * MySQL.
 *
 * Without limiting anything contained in the foregoing, this file,
 * which is part of MySQL Connector/Node.js, is also subject to the
 * Universal FOSS Exception, version 1.0, a copy of which can be found at
 * http://oss.oracle.com/licenses/universal-foss-exception.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License, version 2.0, for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301  USA
 */

'use strict';

/**
 * Result set object returned by each database operation.
 * @alias module:Result
 * @param {Object} state - result details
 * @returns {Result}
 */
function Result (state) {
    state = Object.assign({ generatedDocumentIds: [], rowsAffected: 0, warnings: [] }, state);

    return {
        /**
         * Retrieve the number of documents affected by the operation.
         * @function
         * @name module:Result#getAffectedItemsCount
         * @returns {number} The number of documents.
         */
        getAffectedItemsCount () {
            return this.getAffectedRowsCount();
        },

        /**
         * Retrieve the number of rows affected by the operation.
         * @function
         * @name module:Result#getAffectedRowsCount
         * @returns {number} The number of rows.
         */
        getAffectedRowsCount () {
            return state.rowsAffected;
        },

        /**
         * Retrieve the first <code>AUTO INCREMENT</code> value generated by the operation.
         * @function
         * @name module:Result#getAutoIncrementValue
         * @returns {number} The first value.
         */
        getAutoIncrementValue () {
            return state.generatedInsertId;
        },

        /**
         * Retrieve the list of server-side generated document ids.
         * @function
         * @name module:Result#getGeneratedIds
         * @returns {string[]} The list of ids.
         */
        getGeneratedIds () {
            return state.generatedDocumentIds;
        },

        /**
         * MySQL Warning
         * @typedef {Object} Warning
         * @property {Number} level
         * @property {Number} code
         * @property {String} msg
         */

        /**
         * Retrieve the list of warnings generated on the server.
         * @function
         * @name module:Result#getWarnings
         * @returns {Array.<Warning>} The list of warning objects
         */
        getWarnings () {
            return state.warnings;
        },

        /**
         * Retrieve the number of warnings generated on the server.
         * @function
         * @name module:Result#getWarningsCount
         * @returns {number} The number of warnings.
         */
        getWarningsCount () {
            return state.warnings.length;
        }
    };
}

module.exports = Result;
