require 'fileutils'
require 'rexml/document'
require 'xcodeproj/workspace/file_reference'

module Xcodeproj
  # Provides support for generating, reading and serializing Xcode Workspace
  # documents.
  #
  class Workspace
    # @return [Array<String>] the paths of the projects contained in the
    # @return [Array<FileReference>] the paths of the projects contained in the
    #         workspace.
    #
    attr_reader :file_references
    attr_reader :schemes

    # @param  [Array] file_references @see file_references
    #
    def initialize(*file_references)
      @file_references = file_references.flatten
      @schemes = {}
    end

    #-------------------------------------------------------------------------#

    # Returns a workspace generated by reading the contents of the given path.
    #
    # @param  [String] path
    #         the path of the `xcworkspace` file.
    #
    # @return [Workspace] the generated workspace.
    #
    def self.new_from_xcworkspace(path)
      from_s(File.read(File.join(path, 'contents.xcworkspacedata')), File.expand_path(File.dirname(path)))
    rescue Errno::ENOENT
      new
    end

    #-------------------------------------------------------------------------#

    # Returns a workspace generated by reading the contents of the given
    # XML representation.
    #
    # @param  [String] xml
    #         the XML representation of the workspace.
    #
    # @return [Workspace] the generated workspace.
    #
    def self.from_s(xml, workspace_path = '')
      document = REXML::Document.new(xml)
      file_references = document.get_elements('/Workspace/FileRef').map do |node|
        FileReference.from_node(node)
      end
      instance = new(file_references)
      instance.load_schemes(workspace_path)
      instance
    end

    #-------------------------------------------------------------------------#

    # Adds a new path to the list of the of projects contained in the
    # workspace.
    #
    # @param  [String] projpath
    #         The path of the project to add.
    #
    # @return [void]
    #
    def <<(projpath)
      project_file_reference = Xcodeproj::Workspace::FileReference.new(projpath)
      @file_references << project_file_reference
      load_schemes_from_project File.expand_path(projpath)
    end

    # Checks if the workspace contains the project with the given file
    # reference.
    #
    # @param  [FileReference] file_reference
    #         The file_reference to the project.
    #
    # @return [Boolean] whether the project is contained in the workspace.
    #
    def include?(file_reference)
      @file_references.include?(file_reference)
    end

    # @return [String] the XML representation of the workspace.
    #
    def to_s
      contents = file_references.map { |reference| file_reference_xml(reference) }
      root_xml(contents.join(''))
    end

    # Saves the workspace at the given `xcworkspace` path.
    #
    # @param  [String] path
    #         the path where to save the project.
    #
    # @return [void]
    #
    def save_as(path)
      FileUtils.mkdir_p(path)
      File.open(File.join(path, 'contents.xcworkspacedata'), 'w') do |out|
        out << to_s
      end
    end

    #-------------------------------------------------------------------------#

    # Load all schemes from all projects in workspace
    #
    # @param [String] workspace_dir_path
    #         path of workspaces dir
    #
    # @return [void]
    #
    def load_schemes(workspace_dir_path)
      @file_references.each do |file_reference|
        project_full_path = file_reference.absolute_path(workspace_dir_path)
        load_schemes_from_project(project_full_path)
      end
    end

    private

    # Load all schemes from project
    #
    # @param [String] project_full_path
    #         project full path
    #
    # @return [void]
    #
    def load_schemes_from_project(project_full_path)
      schemes = Xcodeproj::Project.schemes project_full_path
      schemes.each do |scheme_name|
        @schemes[scheme_name] = project_full_path
      end
    end

    # @return [String] The template of the workspace XML as formated by Xcode.
    #
    # @param  [String] contents The XML contents of the workspace.
    #
    def root_xml(contents)
      <<-DOC
<?xml version="1.0" encoding="UTF-8"?>
<Workspace
   version = "1.0">
   #{contents.strip}
</Workspace>
      DOC
    end

    # @return [String] The XML representation of a file reference.
    #
    # @param  [FileReference] file_reference The file reference.
    #
    def file_reference_xml(file_reference)
      <<-DOC
   <FileRef
      location = "#{file_reference.type}:#{file_reference.path}">
   </FileRef>
      DOC
    end

    #-------------------------------------------------------------------------#
  end
end
