'------------------------------------------------------
' Save historic versions of the backup, so that you can
' backtrack to previous versions. This script will, if
' ran BEFORE the SyncBack-backup, copy the last backup
' (either directory or zip-file) to a historic version
' in the same destination directory.
'
' The historic set consists of folders/files in the
' backup-directory that have a "1_", "2_" etc prefixed
' to the original backupfilename or -foldername. After
' the (in this script configured) maximum number of
' historic versions is reached, the script starts over
' at "1_" and repeats the cyclus.
'
' That way, by looking at the modifation-date of the
' historic files or folders, you can always find the
' previous backups. Because of this, you can not
' automatically assume that the "1_" version is the
' oldest or newest version, it all depends on the
' modification-date of the files or folders.
'
' Created by Alex, based on code from timestamp.vbs as
' provided by Michael J. Leaver (www.2BrightSparks.com)
'
' Free for non-commercial use.
'
' Run by calling "cscript rotate.vbs <arg>"
' where <arg> is full path of the destination file or
' folder.
' Files should be full filenames (path included)
' and without any wildcards.
' Folders can either end with a "\" or not.
'
' Examples (for use in the "Run before profile" field
' of the Programs-tab (Expert mode) of SyncBack:
' -- for a file-backup to 'backup.zip' -- :
' cscript "C:\Program Files\2brightsparks\SyncBack\rotate.vbs" "C:\<backupdir>\backUp.zip"
' -- for a folder-backup to subfolder 'backup' of the backup-dir -- :
' cscript "C:\Program Files\2brightsparks\SyncBack\rotate.vbs" "C:\<backupdir>\backup"
'
' All provided that you created rotate.vbs (with this code as content)
' in C:\Program Files\2brightsparks\SyncBack\
'
' By modifying the variable 'nRotationSize', you can
' determine how many historic versions are stored.
'------------------------------------------------------

'set this variable to the number of historic versions you want to preserve.
'eg, if set to 7, the last seven daily backups are stored in the backup directory
nRotationSize = 7

'------------------------------------------------------
'Return the pathname portion of a full pathname
'------------------------------------------------------
Function Pathname(FullPath)
Dim x, y
Dim tmpstring

x = Len(FullPath)
For y = x to 1 step -1
If Mid(FullPath, y, 1) = "\" Or _
Mid(FullPath, y, 1) = "/" Then
tmpstring = Mid(Fullpath, 1, y-1)
Exit For
End If
Next

Pathname = tmpstring

End Function

'------------------------------------------------------
'Return the filename portion of a full pathname
'------------------------------------------------------
Function Basename(FullPath)
Dim x, y
Dim tmpstring

tmpstring = FullPath
x = Len(FullPath)
For y = x To 1 step -1
If Mid(FullPath, y, 1) = "\" Or _
Mid(FullPath, y, 1) = ":" Or _
Mid(FullPath, y, 1) = "/" Then
tmpstring = Mid(Fullpath, y+1)
Exit For
End If
Next
Basename = tmpstring
End Function

'------------------------------------------------------
'Main code
'------------------------------------------------------

Set objArgs = WScript.Arguments
Set fso = CreateObject("Scripting.FileSystemObject")

If (objArgs.Count < 1) then
WScript.Echo "No filename or dirname to copy was supplied."
Else
' Concatenate all the arguments to create on file/dirname
' This is to avoid problems with spaces in filenames or dirnames.
OldFileDirname = ""
For I = 0 To objArgs.Count - 1
OldFileDirname = OldFileDirname & objArgs.Item(I) & " "
Next
OldFileDirname = Trim(OldFileDirname)

'whether the argument represents a file or a folder, eliminate any trailing "\".
If (Right(OldFileDirname,1) = "\") Then
OldFileDirname = Left(OldFileDirname, Len(OldFileDirname) -1)
End If

theFolder = Pathname(OldFileDirname)
Set objFileSystem = CreateObject("Scripting.FileSystemObject")
Set oFolder = objFileSystem.GetFolder(theFolder)

lngMaxIndex = 0
lngNewIndex = 0
dtMaxDate = 0

'determine whether user gave file or folder as argument
If fso.FileExists(OldFileDirname) = True Then
blIsFile = True
blIsFolder = False
ElseIf fso.FolderExists(OldFileDirname) = True Then
blIsFile = False
blIsFolder = True
Else
blIsFile = False
blIsFolder = False
End If

'determine whether to loop through files or folders of the directory
If blIsFile = True Then
Set arrLoop = oFolder.Files
ElseIf blIsFolder = True Then
Set arrLoop = oFolder.SubFolders
Else
WScript.Echo "File or directory <" & OldFileDirname & "> not found."
End If

'start loop and determine the next-to-be-used prefix ("1_" or "2_" etc)
For Each FileOrFolder in arrLoop
For i = 1 to nRotationSize
If Basename(FileOrFolder) = i & "_" & Basename(OldFileDirname) Then
'found index, save it, if max
If FileOrFolder.DateLastModified >= dtMaxDate Then
dtMaxDate = FileOrFolder.DateLastModified
lngMaxIndex = i
End If
Else
'found non-conforming file, skip
End If
Next

If lngMaxIndex > 0 then
'found max index, increase it by 1 to generate the next index for the backups.
'The mod-command makes sure that the indices rotate and do not increase forever.
lngNewIndex = (lngMaxIndex Mod nRotationSize) + 1
Else
'no max index found, start with 1
lngNewIndex = 1
End If
Next

If blIsFile = True Then
'the user gave a file-name as argument
'create the new filename by pre-pending the counter-index to the original filename
'This way, the file 'backup.zip' will be saved as, for example, "1_backup.zip"
NewFilename = Pathname(OldFileDirname) & "\" & lngNewIndex & "_" & Basename(OldFileDirname)

'copy the old file to the new file, so that the old file can be overwritten by SyncBackup after this script is finished.
fso.CopyFile OldFileDirname, NewFilename ,true

WScript.Echo "File <" & NewFilename & "> created."

ElseIf blIsFolder = True Then
'the user gave a folder-name as argument
'create the new foldername by pre-pending the counter-index to the original foldername
'This way, the folder 'backup' will be saved as, for example, "1_backup"
NewFoldername = PathName(OldFileDirname) & "\" & lngNewIndex & "_" & BaseName(OldFileDirname)

If fso.FolderExists(NewFoldername) = True Then
Set oFolder = objFileSystem.GetFolder(NewFoldername)
'do an explicit delete, because the WSH-CopyFolder command doesn't seem to update the DateLastModified-field
'of the destination-folder (either bug or feature).
'That causes a problem when trying to find the next rotation-folder based on the DateLastModified-field.
oFolder.Delete()
End If

'copy the original folder to one of the rotating folders
fso.CopyFolder OldFileDirname, NewFoldername, True

WScript.Echo "Directory <" & NewFoldername & "> created."

Else
WScript.Echo "File or directory <" & OldFileDirname & "> not found."
End If

End If
