// iZoom.cpp
// ---------------------------------------------------------------------------------
// iZoom was created as an example of the berIcon plugin system.  Until we finalize
// the plugin system and properly document it, this example will be your gateway to
// writing you r own plugins for berIcon.

// Some important things to note:
// An option's Group and Name will ALWAYS be passed thru the translation engine
// for the user interface.  This has no effect from the plugin's point-of-view, but
// you should ALWAYS try to use words/phrases that have ALREADY BEEN TRANSLATED
// to ensure that people from all over the world can understand your options.
// To find words that are already translated, look in your language's ini file
// (english="Languages\1033.ini")

// If you're adding a group or option that has not been translated, feel free to
// write it out as more than one word (eg "my option").  Our translation system can
// easily handle translating this later on.  We stick to single-words for easier
// development...

// If you're adding a custom window/interface for your options, remember to use the
// Uber_Translate() function to translate text to display.  the string you pass MUST
// be MAX_PATH characters long to avoid buffer overflows. Remember that "Bonjour" is
// longer than "Hello" :)
//
// Example:
// wchar_t text[MAX_PATH]=L"Hello";
// Uber_Translate(text);//text is now "Bonjour"...

// More about the Options System:
// Each Option has 4 properties.
// 1) Group		-> The name of the option's logical group
// 2) Name		-> The name of the option
// 3) Value		-> A UINT value that is remembered for you...
// 4) sValue	-> A string value that is remembered for you...
//
// The Option GroupModes act as follows:
// UBER_GROUPMODE_NORMAL	->	Normal option...
// UBER_GROUPMODE_HIDDEN	->	Invisible option (simply used for storing info)...
// UBER_GROUPMODE_CHECKBOX  ->	Similar to NORMAL, only a Checkbox is shown next to
//								Options with a non-zero value.
// UBER_GROUPMODE_RADIO		->	Similar to CHECKBOX, only when you set an option's
//								value, all other options in the group reset to 0
//
// ALL MODES call your OptionSelected() procedure
// ---------------------------------------------------------------------------------

#include "stdafx.h"
#include "../../../UberAPI/UberAPI.h"


LRESULT CALLBACK IconProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam);
VOID CALLBACK IconTimerProc(HWND hWnd, UINT uMsg, UINT_PTR idEvent, DWORD dwTime);

Gdiplus::GdiplusStartupInput gdiplusStartupInput;
ULONG_PTR  gdiplusToken;

BOOL APIENTRY DllMain( HANDLE hModule, 
                       DWORD  ul_reason_for_call, 
                       LPVOID lpReserved
					 )
{

	switch(ul_reason_for_call)
	{
	case DLL_PROCESS_ATTACH://Init
		{
			//Plugin Info (Automatically Hidden)
			Uber_AddOption(L"PluginInfo", L"Name", 0, L"iZoom");//Overrides the display name of the Plugin :)
			Uber_AddOption(L"PluginInfo", L"Version", 0, L"1.0");//Might be used in the future
			Uber_AddOption(L"PluginInfo", L"Author", 0, L"PolyVector");//Might be used in the future
			Uber_AddOption(L"PluginInfo", L"URL", 0, L"www.punksoftware.com");//Might be used in the future
			Uber_AddOption(L"PluginInfo", L"Contact", 0, L"support@punksoftware.com");//Might be used in the future

			//Options for the user
			Uber_SetOptionGroupMode(L"PluginSettings", UBER_GROUPMODE_CHECKBOX);
			Uber_AddOption(L"PluginSettings", L"Echo", true, L"");
			Uber_AddOption(L"PluginSettings", L"Outline", false, L"");
			Uber_AddOption(L"PluginSettings", L"Async", false, L"");

			Uber_AddOption(L"", L"AboutPlugin", 0, L"");


			//The following is a demonstration of the HIDDEN GroupType...
			//The Option "FirstTime" is created in a hidden group and
			//is set to 'true' by default...
			//After the plugin is run for the first time it's set to false
			Uber_SetOptionGroupMode(L"Hidden", UBER_GROUPMODE_HIDDEN);
			Uber_AddOption(L"Hidden",L"FirstTime",true,L"");
			if(Uber_GetOption(L"Hidden",L"FirstTime"))
			{
				//MessageBox(0,L"This is the first time this plugin has been loaded",L"Notice",0);
				Uber_SetOption(L"Hidden",L"FirstTime",false);
			}
			

			//Icon Zoomer Class
			WNDCLASSEX wcex;
			wcex.cbSize = sizeof(WNDCLASSEX); 
			wcex.style			= CS_HREDRAW | CS_VREDRAW;
			wcex.lpfnWndProc	= (WNDPROC)IconProc;
			wcex.cbClsExtra		= 0;
			wcex.cbWndExtra		= 0;
			wcex.hInstance		= NULL;
			wcex.hIcon			= NULL;
			wcex.hCursor		= NULL;
			wcex.hbrBackground	= NULL;
			wcex.lpszMenuName	= _T("");
			wcex.lpszClassName	= _T("iZoomWnd");
			wcex.hIconSm		= NULL;
			RegisterClassEx(&wcex);

			// Initialize Gdiplus
			GdiplusStartup(&gdiplusToken, &gdiplusStartupInput, NULL);

		}

		break;
	case DLL_PROCESS_DETACH://Exit
		//MessageBox(0,L"Plugin Exit",L"",0);

		// Shutodwn Gdiplus
		Gdiplus::GdiplusShutdown(gdiplusToken);
		break;
	}
    return TRUE;
}

//OptionSelected()
//-------------------------------------------------------
//This function is called when an UBER_GROUPMODE_CALLBACK
//Option is clicked on.
//-------------------------------------------------------
bool UBERFUNCTION OptionSelected(wchar_t* Group, wchar_t* Name)
{
	//MessageBox(0,Name,Group,0);
	if(!wcscmp(Group,L""))
	{
		if(!wcscmp(Name,L"AboutPlugin"))
		{
			wchar_t About[MAX_PATH]=L"AboutPlugin";
			Uber_Translate(About);
			MessageBox(0,L"iZoom was created by PunkSoftware\n\nIf you are interested in developing Effects\nlook in the 'Plugins' folder for sourcecode.",About,0);
		}
	}
	return true;
}

//ActivateIcon()
//---------------------------------------------------------
//This function gets called each time an Icon is Activated
//Simply start your ber cool special effect here :)
//Icon execution is blocked until this function returns
//
//A simple SetTimer() or CreateThread() call will allow for
//async special effects :)
//---------------------------------------------------------
bool UBERFUNCTION ActivateIcon(UBER_ICON* data)
{
	//IMPORTANT NOTES:
	//Create the window invisible and later show it with SW_SHOWNOACTIVATE
	//This will prevent focus from leaving the current window.
	//Using WS_EX_TRANSPARENT will prevent users clicking on the effect's window
	HWND hWnd=CreateWindowEx(WS_EX_TOPMOST| WS_EX_LAYERED | WS_EX_TRANSPARENT | WS_EX_TOOLWINDOW, _T("iZoomWnd"), _T(""), WS_POPUP | WS_CLIPSIBLINGS, 0, 0, 64, 64, NULL, NULL, NULL, NULL);
	ShowWindow(hWnd,SW_SHOWNOACTIVATE);


	Gdiplus::Bitmap* IconBitmapCopy=NULL;

	//Creates a Bitmap of the Outline of the Icon
	if(Uber_GetOption(L"PluginSettings", L"Outline"))
	{
		IconBitmapCopy=new Gdiplus::Bitmap(data->bitmap->GetWidth(),data->bitmap->GetHeight(),PixelFormat32bppARGB);
		int width=data->bitmap->GetWidth();
		int height=data->bitmap->GetHeight();
		for(int y=0;y<height;y++)
		{
			for(int x=0;x<width;x++)
			{
				Gdiplus::Color cMid;
				data->bitmap->GetPixel(x,y,&cMid);
				if(cMid.GetAlpha())
				{
					if(x==0||y==0||x==width-1||y==height-1)
						IconBitmapCopy->SetPixel(x,y,Color(255,0,0,0));
					else
					{
						Gdiplus::Color cLeft,cTop,cRight,cBottom;
						data->bitmap->GetPixel(x-1,y,&cLeft);
						data->bitmap->GetPixel(x,y-1,&cTop);
						data->bitmap->GetPixel(x+1,y,&cRight);
						data->bitmap->GetPixel(x,y+1,&cBottom);
						if(!cLeft.GetAlpha()||!cTop.GetAlpha()||!cRight.GetAlpha()||!cBottom.GetAlpha())
							IconBitmapCopy->SetPixel(x,y,Color(255,0,0,0));
					}
				}


			}
		}

	}
	else
	{
		IconBitmapCopy=data->bitmap->Clone(0,0,data->bitmap->GetWidth(),data->bitmap->GetHeight(),PixelFormat32bppARGB);
	}

	SetProp(hWnd,_T("IcoBmp"),(HANDLE)IconBitmapCopy);
	RECT ItemRect=data->rect;
	SetProp(hWnd,_T("midX"),(HANDLE)((ItemRect.right+ItemRect.left)/2));//MiddleX
	SetProp(hWnd,_T("midY"),(HANDLE)((ItemRect.bottom+ItemRect.top)/2));//MiddleY
	int minSize=(ItemRect.bottom-ItemRect.top);
	SetProp(hWnd,_T("minSize"),(HANDLE)minSize);//MiddleY
	int SysIconSize=GetSystemMetrics(SM_CXICON);
	int maxSize=SysIconSize*8;
	if(maxSize<minSize+(SysIconSize*2))
		maxSize=minSize+(SysIconSize*2);
	SetProp(hWnd,_T("maxSize"),(HANDLE)maxSize);//MiddleY
	SetProp(hWnd,_T("ms"),(HANDLE)0);

	if(Uber_GetOption(L"PluginSettings", L"Async"))
	{
		//animate fx while loading (sometimes ugly)
		SetTimer(hWnd,NULL,1000/75,IconTimerProc);
	}
	else
	{
		//Wait for fx to finish before continuing (smoooooth)
		while(IsWindow(hWnd))
			IconTimerProc(hWnd,0,0,0);
	}
	return true;
}

LRESULT CALLBACK IconProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	return DefWindowProc(hWnd,message,wParam,lParam);
}

VOID CALLBACK IconTimerProc(HWND hWnd, UINT uMsg, UINT_PTR idEvent, DWORD dwTime)
{
			int OldMS=(int)GetProp(hWnd,_T("ms"));
			if(!OldMS)
			{
				OldMS=GetTickCount();
				SetProp(hWnd,_T("ms"),(HANDLE)OldMS);
			}
			int NewMS=GetTickCount();
			Gdiplus::Bitmap* IconBitmap=(Gdiplus::Bitmap*)GetProp(hWnd,_T("IcoBmp"));
			if(IconBitmap)
			{
				int MinSize=(int)GetProp(hWnd,_T("minSize"));
				int MaxSize=(int)GetProp(hWnd,_T("maxSize"));
				int IconSize=MinSize+int(float(NewMS-OldMS)/200.0f * (MaxSize-MinSize));
				float alpha=(float(MaxSize-IconSize)/float(MaxSize-MinSize));
				alpha*=0.55f;
				if(alpha<0.0f)
					alpha=0.0f;
				POINT WindowOrg;
				WindowOrg.x=(int)GetProp(hWnd,_T("midX"))-(IconSize/2);
				WindowOrg.y=(int)GetProp(hWnd,_T("midY"))-(IconSize/2);
				
				//Setup Buffer
				HBITMAP BufferBM;
				HDC BufferDC;
				BITMAPINFO BufferBMInfo;
				ZeroMemory(&BufferBMInfo,sizeof(BITMAPINFO));
				BufferBMInfo.bmiHeader.biSize=sizeof(BITMAPINFOHEADER);
				BufferBMInfo.bmiHeader.biWidth=IconSize;
				BufferBMInfo.bmiHeader.biHeight=IconSize;
				BufferBMInfo.bmiHeader.biPlanes=1;
				BufferBMInfo.bmiHeader.biBitCount=32;
				BufferDC=CreateCompatibleDC(NULL);
				BufferBM=CreateDIBSection(BufferDC,&BufferBMInfo,DIB_RGB_COLORS,NULL,NULL,NULL);
				SelectObject(BufferDC,BufferBM);
				Gdiplus::Graphics graphics(BufferDC);
				graphics.SetCompositingMode(CompositingModeSourceCopy);
				graphics.SetCompositingQuality(CompositingQualityHighSpeed);
				graphics.SetInterpolationMode(InterpolationModeBilinear);

				//Start Rendering
				int TrailCount=1;
				if(Uber_GetOption(L"PluginSettings", L"Echo"))
					TrailCount=5;
				int IconSizeOffset=0;
				for(float i=0;i<TrailCount;i++)
				{
					IconSizeOffset=((IconSize-IconBitmap->GetWidth())/TrailCount)*(i);
					graphics.DrawImage(	IconBitmap,
						Gdiplus::Rect(IconSizeOffset/2,IconSizeOffset/2,IconSize-IconSizeOffset,IconSize-IconSizeOffset),
						0,
						0,
						IconBitmap->GetWidth(),
						IconBitmap->GetHeight(),
						Gdiplus::UnitPixel,
						NULL);
					graphics.SetCompositingMode(CompositingModeSourceOver);
				}

				SIZE BufferSize={IconSize,IconSize};
				POINT SrcOrg={0,0};
				BLENDFUNCTION Blend;
				Blend.BlendOp=AC_SRC_OVER;
				Blend.BlendFlags=0;
				Blend.SourceConstantAlpha=BYTE(255*alpha);
				Blend.AlphaFormat=AC_SRC_ALPHA;

				UpdateLayeredWindow(hWnd,0,&WindowOrg,&BufferSize,BufferDC,&SrcOrg,RGB(0,0,0),&Blend,ULW_ALPHA);

				DeleteDC(BufferDC);
				DeleteObject(BufferBM);
				if(alpha==0.0f)
				{
					KillTimer(hWnd,NULL);
					ShowWindow(hWnd,SW_HIDE);
					DestroyWindow(hWnd);
					SetProp(hWnd,_T("IcoBmp"),NULL);
					delete IconBitmap;
				}
			}
}