--[[
Copyright 2008, 2009, 2010, 2011 João Cardoso
Sushi is distributed under the terms of the GNU General Public License (or the Lesser GPL).
This file is part of Sushi.

Sushi is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Sushi is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Sushi. If not, see <http://www.gnu.org/licenses/>.
--]]

local Sake = LibStub('Sake-1.0')
local Sushi = LibStub('Sushi-2.0')

local Container, old = Sushi:NewWidgetClass('Container', {
	frameType = 'ScrollFrame',
	template = 'UIPanelScrollFrameTemplate',
	insets = {top = 10, bottom = 10, left = 15, right = 15},
	version = 2,
})

if not Container then
	return
end


--[[ Locals ]]--

local empty, total = {}, {}
local Insets = {'top', 'bottom', 'left', 'right'}

local DefaultBackdrop = {
	bgFile = 'Interface\\ChatFrame\\ChatFrameBackground',
	edgeFile = 'Interface\\Tooltips\\UI-Tooltip-Border',
	tile = true, tileSize = 16, edgeSize = 16,
	insets = {left = 4, right = 4, top = 4, bottom = 4},
	bgColor = {0.1, 0.1, 0.1, 0.5},
	edgeColor = {0.4, 0.4, 0.4},
}


--[[ Events ]]--

function Container:OnCreate()
	local name = self:GetName()
	local ScrollBar = _G[name..'ScrollBar']
	ScrollBar:ClearAllPoints()
	
	local TopButton = _G[name..'ScrollBarScrollUpButton']
	TopButton:SetPoint('BOTTOM', ScrollBar, 'TOP', 0, -4)
	
	local DownButton = _G[name..'ScrollBarScrollDownButton']
	DownButton:SetPoint('TOP', ScrollBar, 'BOTTOM', 0, 4)
	
	local ScrollBorder = ScrollBar:CreateTexture(nil, 'BACKGROUND')
	ScrollBorder:SetTexCoord(0.125, 0.250, 0, 1)
	
	local BackdropFrame = CreateFrame('Frame', nil, self)
	BackdropFrame:SetFrameLevel(0)
	
	local metatable = getmetatable(self).__index
	self.SetRealHeight = metatable.SetHeight
	self.SetRealWidth = metatable.SetWidth
	
	self.ScrollBar = ScrollBar
	self.TopButton = TopButton
	self.DownButton = DownButton
	self.ScrollBorder = ScrollBorder
	self.BackdropFrame = BackdropFrame
	self.scrollBarHideable = true
	self.children = {}
	
	self:SetScrollChild(CreateFrame('Frame', nil, self))
	self:EnableMouseWheel(nil)
	
	self:HookScript('OnScrollRangeChanged', self.OnScrollRangeChanged)
	self:SetScript('OnSizeChanged', self.OnSizeChanged)
	self:SetScript('OnMouseWheel', self.OnMouseWheel)
	self:GetScript('OnScrollRangeChanged')(self)
	
	self:SetScript('OnHide', self.ReleaseChildren)
	self:SetScript('OnShow', self.UpdateChildren)
end

function Container:OnAcquire()
	self:SetBackdrop(nil)
	self:SetHeight(nil)
	self:SetWidth(nil)
end

function Container:OnChildUpdate()
	self:UpdateChildren()
	self:FireCallback('OnUpdate')
end

function Container:OnRelease()
	self:SetChildren(nil)
	self:SetLayout(nil)
end


--[[ Scripts ]]--

function Container:OnMouseWheel(value)
	local ScrollBar = self.ScrollBar
	ScrollBar:SetValue(ScrollBar:GetValue() - value * self:GetScrollChild():GetHeight() / 50)
end

function Container:OnScrollRangeChanged()
	local enableScroll = floor(self:GetVerticalScrollRange()) > 0 or nil
	local enabled = self:IsMouseWheelEnabled() and true
	
	if enableScroll ~= enabled then
		self:OnSizeChanged()
		
		if enableScroll then
			self:EnableMouseWheel(true)
		else
			self:EnableMouseWheel(nil)
		end
	end
end

function Container:OnSizeChanged()
	self:UpdateLayout()
	self:UpdateChildren()
end


--[[ Width ]]--

function Container:SetWidth(width)
	self.width = width
	self:UpdateWidth()
end

function Container:UpdateWidth()
	local width = self.width or self.layoutWidth
	if width == true then
		local parent = self:GetParent()
		width = parent and parent:GetWidth() - 30
	end
	
	self:SetRealWidth(width or 0)
end


--[[ Height ]]--

function Container:SetHeight(height)
	self.height = height
	self:UpdateHeight()
end

function Container:UpdateHeight()
	local height = self.height or self.layoutHeight
	if height == true then
		local parent = self:GetParent()
		height = parent and parent:GetHeight() - 20
	end
	
	self:SetRealHeight(height or 0)
end


--[[ Backdrop ]]--

function Container:SetBackdrop(backdrop)
	self.backdrop = backdrop
	if backdrop then
		if backdrop == true then
			backdrop = DefaultBackdrop
		end
		
		local BackdropFrame = self.BackdropFrame
		local insets = backdrop.insets or empty
		local edgeSize = backdrop.edgeSize or 0
		local size = edgeSize / 16
		
		BackdropFrame:SetBackdrop(backdrop)
		BackdropFrame:SetBackdropColor(unpack(backdrop.bgColor or empty))
		BackdropFrame:SetBackdropBorderColor(unpack(backdrop.edgeColor or empty))
		
		for _,k in pairs(Insets) do
			local off = insets[k] or 0
	
			if k == 'left' then
				off = - off
			elseif k == 'bottom' then
				off = - (off + 2 - size)
			end
			
			BackdropFrame:SetPoint(k, self, k, off, off)
		end
		
		local ScrollBar = self.ScrollBar
		local ScrollBorder = self.ScrollBorder
		local left = 4 + size * 11 + ScrollBar:GetWidth()
		local right = size * 2 - 3
		
		ScrollBar:SetPoint('BOTTOMRIGHT', right, 10.4 - size / 5)
		ScrollBar:SetPoint('TOPRIGHT', right, size * 1.7 - 13.7)
		
		ScrollBorder:SetPoint('TOPLEFT', self, 'TOPRIGHT', -left, size * 2 - 2)
		ScrollBorder:SetPoint('BOTTOMLEFT', self, 'BOTTOMRIGHT', -left, -1)
		ScrollBorder:SetVertexColor(BackdropFrame:GetBackdropBorderColor())
		ScrollBorder:SetTexture(backdrop.edgeFile)
		ScrollBorder:SetWidth(edgeSize)
	else
		self.ScrollBar:SetPoint('BOTTOMRIGHT', -3, 10.4)
		self.ScrollBar:SetPoint('TOPRIGHT', -3, -13.7)
		self.BackdropFrame:SetBackdrop(nil)
		self.ScrollBorder:SetTexture(nil)
	end
end

function Container:GetBackdrop()
	return self.backdrop
end


--[[ Children ]]--

function Container:SetChildren(...)
	if ... then
		self:RegisterCallback('CreateChildren', ...)
	else
		self:UnregisterCallback('CreateChildren')
	end
	
	if self:IsVisible() then
		self:UpdateChildren()
	end
end

function Container:UpdateChildren()
	self:ReleaseChildren()
	self:FireCallback('CreateChildren')
	
	for i, child in self:IterateChildren() do
		local top, bottom, left, right = self:GetChildInsets(child)
		local class = child.frameCache
	
		self:RegisterLayoutChild(child, {
			height = class.height or child:GetHeight(),
			width = class.width or child:GetWidth(),
			bottom = bottom,
			right = right,
			left = left,
			top = top,
		})
	end
	
	self:UpdateLayout()
end

function Container:ReleaseChildren()
	for i, child in self:IterateChildren() do
		child.UnregisterCallback(self, 'OnUpdate')
		child.insets = nil
		child:Release()
	end
	
	self:UnregisterLayoutChildren()
	wipe(self.children)
end

function Container:IterateChildren()
	return pairs(self.children)
end

function Container:GetNumChildren()
	return #self.children
end


--[[ Child ]]--

function Container:CreateChild(class)
	local child = Sushi:CreateWidget(class, self:GetScrollChild())
	child.RegisterCallback(self, 'OnUpdate', 'OnChildUpdate')
	
	tinsert(self.children, child)
	return child
end

function Container:GetChild(i)
	return self.children[i]
end

function Container:GetChildInsets(child)
	local class = child.frameCache.insets or empty
	local child = child.insets or empty
	local total = wipe(total)

	for i,k in pairs(Insets) do
		total[i] = (child[i] or child[k] or 0) + (class[k] or  0)
	end
	
	return unpack(total)
end


--[[ Layout ]]--

function Container:SetLayout(...)
	self.layout, self.layoutData = ...
	self:UpdateLayout()
end

function Container:UpdateLayout()
	local layout, data = self:GetLayout()
	if layout then
		local ScrollChild = self:GetScrollChild()
		local ScrollBar = self.ScrollBar
		
		data = data or {}
		data.maxWidth = (data.maxWidth or self:GetWidth()) - (ScrollBar:IsShown() and ScrollBar:GetWidth() + 6 or 0)
		data.anchor = data.anchor or ScrollChild
		
		local width, height = self:PerformLayout(layout, data)
		ScrollChild:SetHeight(height or self:GetHeight() - 20)
		ScrollChild:SetWidth(width or data.maxWidth)
		
		self.layoutWidth = width
		self.layoutHeight = height
		self:UpdateHeight()
		self:UpdateWidth()
	end
end

function Container:GetLayout()
	return self.layout, self.layoutData
end


--[[ Embedding ]]--

if not old then
	Sake:Embed(Container, 'GetLayout', 'GetSortation')
else
	Container:UpdateEmbeds()
end