--[[
Copyright 2008, 2009, 2010, 2011 João Cardoso
Sushi is distributed under the terms of the GNU General Public License (or the Lesser GPL).
This file is part of Sushi.

Sushi is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Sushi is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Sushi. If not, see <http://www.gnu.org/licenses/>.
--]]

local Sushi = LibStub('Sushi-2.0')
local Dropdown = Sushi:NewWidgetClass('Dropdown', {
	frameType = 'Frame',
	template = 'UIDropDownMenuTemplate',
	insets = {top = 17, bottom = 3, left = -3},
	version = 3,
})

if not Dropdown then
	return
end


--[[ Startup ]]--

function Dropdown:OnCreate()
	local name = self:GetName()
	
	local Label = self:CreateFontString()
	Label:SetPoint('BOTTOMRIGHT', self, 'TOPRIGHT', -16, 3)
	Label:SetPoint('BOTTOMLEFT', self, 'TOPLEFT', 16, 3)
	Label:SetJustifyH('Left')
	
	local Right = _G[name .. 'Right']
	Right:ClearAllPoints()
	Right:SetPoint('TOPRIGHT', 0, 17)
	
	local Middle = _G[name .. 'Middle']
	Middle:SetPoint('RIGHT', Right, 'LEFT')
	
	self:SetScript('OnEnter', self.OnEnter)
	self:SetScript('OnLeave', self.OnLeave)
	
	self.GetValue = UIDropDownMenu_GetSelectedValue
	self.Button = _G[name .. 'Button']
	self.Value = _G[name .. 'Text']
	self.Label = Label
	
	self.linesOrder = {}
	self.linesNames = {}
	self.linesDesc = {}
	
	UIDropDownMenu_Initialize(self, self.Initialize)
end

function Dropdown:OnAcquire()
	self:SetWidth(190)
	self:UpdateFonts()
end

function Dropdown:OnRelease()
	self.small, self.desc = nil
	self:SetDisabled(nil)
	self:SetLabel(nil)
	self:SetValue(nil)
	self:ClearLines()
end


--[[ Label ]]--

function Dropdown:SetLabel(label)
	self.Label:SetText(label)
end

function Dropdown:GetLabel()
	return self.Label:GetText()
end


--[[ Font ]]--

function Dropdown:SetDisabled(disabled)
	if disabled then
		self.Button:Disable()
		self:EnableMouse(nil)
	else
		self.Button:Enable()
		self:EnableMouse(true)
	end
	
	self.disabled = disabled
	self:UpdateFonts()
end

function Dropdown:IsDisabled()
	return self.disabled
end

function Dropdown:SetSmall(small)
	self.small = small
	self:UpdateFonts()
end

function Dropdown:IsSmall()
	return self.small
end

function Dropdown:UpdateFonts()
	local font
	if not self:IsDisabled() then
		font = 'GameFont%s'
	else
		font = 'GameFontDisable'
	end
	
	self.Label:SetFontObject(font:format('Normal') .. (self:IsSmall() and 'Small' or ''))
	self.Value:SetFontObject(font:format('Highlight') .. 'Small')
end


--[[ Initialize ]]--

function Dropdown:SetValue(value)
	UIDropDownMenu_SetSelectedValue(self, value)
	self.Value:SetText(self.linesNames[value])
end

function Dropdown:Initialize()
	for value, name, desc in self:IterateLines() do
		local line = UIDropDownMenu_CreateInfo()
		line.checked = self:GetValue() == value
		line.tooltipTitle = desc and name
		line.tooltipText = desc
		line.value = value
		line.text = name
		
		line.func = function()
			if value ~= self:GetValue() then
				self:FireCallback('OnValueChanged', value)
				self:FireCallback('OnLineSelected', value)
				self:FireCallback('OnUpdate')
			end
		end
		
		UIDropDownMenu_AddButton(line)
	end
end


--[[ Lines ]]--

function Dropdown:AddLine(value, name, desc)
	if not self.linesNames[value] then
		tinsert(self.linesOrder, value)
	end
	
	self.linesNames[value] = name or value
	self.linesDesc[value] = desc
	
	if value == self:GetValue() then
		self:SetValue(value)
	end
end

function Dropdown:GetLine(value)
	return self.linesNames[value], self.linesDesc[value]
end

function Dropdown:IterateLines()
	local i, value, name, desc = 1
	return function()
		value = self.linesOrder[i]
		i = i + 1
		
		return value, self:GetLine(value)
	end
end
	
function Dropdown:ClearLines()
	wipe(self.linesOrder)
	wipe(self.linesNames)
	wipe(self.linesDesc)
end


--[[ Tooltip ]]--

function Dropdown:SetDescription(desc)
	self.desc = desc
end

function Dropdown:GetDescription()
	return self.desc
end

function Dropdown:OnEnter()
	local desc = self:GetDescription()
	if desc then
		GameTooltip:SetOwner(self, 'ANCHOR_TOPRIGHT')
		GameTooltip:SetText(desc, nil, nil, nil, nil, 1)
		GameTooltip:Show()
	end
end

function Dropdown:OnLeave()
	if self:GetDescription() then
		GameTooltip:Hide()
	end
end

Dropdown:UpdateEmbeds()