--[[
Copyright 2008, 2009, 2010, 2011 João Cardoso
Scrap is distributed under the terms of the GNU General Public License (or the Lesser GPL).
This file is part of Scrap.

Scrap is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Scrap is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Scrap. If not, see <http://www.gnu.org/licenses/>.
--]]

local Tooltip = CreateFrame('GameTooltip', 'ScrapTooltip', nil, 'GameTooltipTemplate')
local Scrap = CreateFrame('Button', 'Scrap', MerchantBuyBackItem)
local Unfit = LibStub('Unfit-1.0')
local L = {}

Scrap_NotJunk = Scrap_NotJunk or {}
Scrap_Junk = Scrap_Junk or {}
Scrap_Locals = L


--[[ Locals ]]--

local WEAPON, ARMOR, _, CONSUMABLES = GetAuctionItemClasses()
local FISHING_ROD = select(17 , GetAuctionItemSubClasses(1))

local CLASS_NAME = LOCALIZED_CLASS_NAMES_MALE[select(2, UnitClass('player'))]
local CAN_TRADE = BIND_TRADE_TIME_REMAINING:format('.*')
local CAN_REFUND = REFUND_TIME_REMAINING:format('.*')
local MATCH_CLASS = ITEM_CLASSES_ALLOWED:format('')
local IN_SET = EQUIPMENT_SETS:format('.*')

local ACTUAL_SLOTS = {
	ROBE = 'CHEST',
	CLOAK = 'BACK',
	RANGEDRIGHT = 'RANGED',
	THROWN = 'RANGED',
	WEAPONMAINHAND = 'MAINHAND',
	WEAPONOFFHAND = 'OFFHAND',
	HOLDABLE = 'OFFHAND',
	SHIELD = 'OFFHAND',
}

local function GetLine(i)
  local line = _G['ScrapTooltipTextLeft'..i]
  return line and line:GetText() or ''
end


--[[ Events ]]--

function Scrap:Startup()
	self:SetScript('OnEvent', function(self, event) self[event](self) end)
	self:RegisterEvent('MERCHANT_SHOW')
end

function Scrap:MERCHANT_SHOW()
	if not LoadAddOn('Scrap_Merchant') then
		self:UnregisterEvent('MERCHANT_SHOW')
	else
		self:MERCHANT_SHOW()
	end
end


--[[ Junk List ]]--

function Scrap:IsJunk(id, bag, slot)
	if id and not Scrap_NotJunk[id] then
		return Scrap_Junk[id] or self:CheckFilters(id, bag, slot)
	end
end

function Scrap:IterateJunk()
	local bagNumSlots, bag, slot = GetContainerNumSlots(BACKPACK_CONTAINER), BACKPACK_CONTAINER, 0
	local match, id
	
	return function()
		match = nil
		
		while not match do
			if slot < bagNumSlots then
				slot = slot + 1
			elseif bag < NUM_BAG_FRAMES then
				bag = bag + 1
				bagNumSlots = GetContainerNumSlots(bag)
				slot = 1
			else
				bag, slot = nil
				break
			end
			
			id = GetContainerItemID(bag, slot)
			match = self:IsJunk(id, bag, slot)
		end
		
		return bag, slot, id
	end
end

function Scrap:ToggleJunk(id)
  if id then
    local isJunk = self:IsJunk(id)

    if Scrap_Junk[id] or Scrap_NotJunk[id] then
      Scrap_Junk[id], Scrap_NotJunk[id] = nil

    elseif isJunk then
      Scrap_NotJunk[id] = true
    else
      Scrap_Junk[id] = true
    end

    local itemLink = select(2, GetItemInfo(id))
    if isJunk then
      self:Print(L.Removed, itemLink, 'LOOT')
    else
      self:Print(L.Added, itemLink, 'LOOT')
    end
  end
end


--[[ Filters ]]--

function Scrap:CheckFilters(id, ...)
	local _, link, quality, level, _, class, subClass, _, slot, _, value = GetItemInfo(id)
	if value and value > 0 then
		if quality == ITEM_QUALITY_POOR then
			return true
		
		elseif Scrap_LowConsume and class == CONSUMABLES then
			 return level ~= 0 and (UnitLevel('player') - level) > 10
			
		elseif class == ARMOR or class == WEAPON then
			if isPoor then
				return level > 10 or UnitLevel('player') > 20
			else
				local isEnchanter = self:IsEnchanter()
				local numLines, limit = self:IsSoulbound(link, ...)
				if numLines then
					return not isEnchanter and (Unfit:IsClassUnusable(subClass, slot) or self:IsOtherClass(numLines, limit)) or self:IsLowEquip(id, subClass, slot, level, quality)
				end
			end
		end
	end
end

function Scrap:IsSoulbound(link, bag, slot)
	Tooltip:SetOwner(UIParent, 'ANCHOR_NONE')
	
	if bag and slot then
		Tooltip:SetBagItem(bag, slot)
	else
		Tooltip:SetHyperlink(link)
	end

	local numLines = Tooltip:NumLines()
	if not GetLine(numLines - 1):match(IN_SET) then
		local lastLine = GetLine(numLines)
		local soulbound = bag and slot and ITEM_SOULBOUND or ITEM_BIND_ON_PICKUP
	
		if not lastLine:match(CAN_TRADE) and not lastLine:match(CAN_REFUND) then
			for i = 2,4 do
				if GetLine(i) == soulbound then
					return numLines, i
				end
			end
		end
	end
end

function Scrap:IsEnchanter()
    local prof1, prof2 = GetProfessions()
    return not prof1 or not prof2 or select(7, GetProfessionInfo(prof1)) == 333 or select(7, GetProfessionInfo(prof2)) == 333
end

function Scrap:IsOtherClass(numLines, limit)
	for i = numLines, limit, -1 do
		local text = GetLine(i)
		if text:match(MATCH_CLASS) then
			return not text:match(CLASS_NAME)
		end
	end
end

function Scrap:IsLowEquip(id, subClass, slot, ...)
	if slot ~= '' and subClass ~= FISHING_ROD then
		slot = slot:match('INVTYPE_(.+)')

		if slot ~= 'TRINKET' and slot ~= 'TABARD' and slot ~= 'BODY' then
			return self:HasBetterEquip(id, slot, ...)
		end
	end
end

function Scrap:HasBetterEquip(id, slot, level, quality)
	if Scrap_LowEquip then
		local slot1, slot2 = ACTUAL_SLOTS[slot] or slot
		local value = level * quality ^.35
		local double
		
		if slot1 == 'WEAPON' or slot1 == '2HWEAPON' then
			if slot1 == '2HWEAPON' then
				double = true
			end
			
			slot1, slot2 = 'MAINHAND', 'OFFHAND'
		elseif slot1 == 'FINGER' then
			slot1, slot2 = 'FINGER1', 'FINGER2'
		end
		
		return self:IsBetterEquip(slot1, value) and (not slot2 or self:IsBetterEquip(slot2, value, double))
	end
end

function Scrap:IsBetterEquip(slot, value, empty)
	local equipedItem = GetInventoryItemID('player', _G['INVSLOT_'..slot])
	if equipedItem then
		local _,_, equipQuality, equipLevel = GetItemInfo(equipedItem)
		return equipLevel * equipQuality^.35 - value > 15
	elseif empty then
		return true
	end
end


--[[ Utiliy ]]--

function Scrap:Print(pattern, value, channel)
  local channel = 'CHAT_MSG_'..channel
  for i = 1, 10 do
    local frame = _G['ChatFrame'..i]
    if frame:IsEventRegistered(channel) then
      ChatFrame_MessageEventHandler(frame, channel, pattern:format(value), nil, nil, '', nil, nil, nil, nil, nil, nil, nil, '')
    end
  end
end

Scrap:Startup()